<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\OrderInventory\Handler;

use Lifestyle\Pimcore\OrderInventory\Configuration\ProductConfiguration;
use Lifestyle\Pimcore\OrderInventory\Model\ProductMapper;
use Lifestyle\Pimcore\OrderInventory\Notifier\OnHandLimitNotifier;
use Pimcore\Model\DataObject;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class SellEventHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\OrderInventory\Handler
 */
class SellEventHandler extends AbstractHandler
{
    /**
     * @var ProductConfiguration
     */
    private $productConfiguration;

    /**
     * @var OnHandLimitNotifier
     */
    private $notifier;

    /**
     * SellEventHandler constructor.
     * @param ProductConfiguration $productConfiguration
     * @param OnHandLimitNotifier $notifier
     * @param ProductMapper $mapper
     * @param ValidatorInterface $validator
     */
    public function __construct(
        ProductConfiguration $productConfiguration,
        OnHandLimitNotifier $notifier,
        ProductMapper $mapper,
        ValidatorInterface $validator
    ) {
        $this->productConfiguration = $productConfiguration;
        $this->notifier = $notifier;
        parent::__construct($mapper, $validator);
    }

    /**
     * @return string
     */
    protected function getResponsibleEvent(): string
    {
        return 'product.inventory.sell';
    }

    /**
     * @param DataObject\AbstractObject $product
     * @param int $changedQuantity
     * @param int|null $onHold
     * @return bool
     */
    protected function execute(DataObject\AbstractObject $product, int $changedQuantity, ?int $onHold): bool
    {
        $onHandProperty = $this->productConfiguration->getStockPropertyName();
        $previousOnHand = $product->get($onHandProperty);
        $product->set($onHandProperty, $previousOnHand - $changedQuantity);

        $onHoldProperty = $this->productConfiguration->getStockOnHoldPropertyName();
        if (null !== $onHold) {
            $product->set($onHoldProperty, $onHold);
        } else {
            // For BC
            $product->set($onHoldProperty, $product->get($onHoldProperty) - $changedQuantity);
        }

        $product->save(['versionNote' => 'Auto update by inventory-sell-process.']);

        $this->notifier->onSell($product, $previousOnHand, $product->get($onHandProperty));

        return true;
    }
}
