<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2018 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Lifestyle\Pimcore\RabbitMQBundle\EventListener;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\RabbitMQBundle\Event\PreUnpublishEvent;
use Lifestyle\Pimcore\RabbitMQBundle\Exception\DeleteNotPossibleException;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;

/**
 * Class PreUnpublishListener
 * @package Lifestyle\Pimcore\RabbitMQBundle\EventListener
 */
class PreUnpublishListener
{
    /**
     * @var HandlerInterface
     */
    private $syliusTaxonHandler;

    /**
     * @var HandlerInterface
     */
    private $syliusProductHandler;

    /**
     * @var HandlerInterface
     */
    private $syliusVariantHandler;

    /**
     * @var HandlerInterface
     */
    private $syliusOptionHandler;

    /**
     * @param HandlerInterface $syliusTaxonHandler
     * @param HandlerInterface $syliusProductHandler
     * @param HandlerInterface $syliusVariantHandler
     * @param HandlerInterface $syliusOptionHandler
     */
    public function __construct(
        HandlerInterface $syliusTaxonHandler,
        HandlerInterface $syliusProductHandler,
        HandlerInterface $syliusVariantHandler,
        HandlerInterface $syliusOptionHandler
    ) {
        $this->syliusTaxonHandler = $syliusTaxonHandler;
        $this->syliusProductHandler = $syliusProductHandler;
        $this->syliusVariantHandler = $syliusVariantHandler;
        $this->syliusOptionHandler = $syliusOptionHandler;
    }

    /**
     * @param PreUnpublishEvent $event
     * @throws DeleteNotPossibleException
     */
    public function onPreUnpublish(PreUnpublishEvent $event)
    {
        $collector = $event->getCollector();

        // First, send all DELETE requests synchronously in order to know if it is possible at all.
        switch ($collector->getItemValue('event')) {
            case 'taxon.unpublish':
                $this->deleteSyliusTaxon($collector);
                break;
            case 'product.unpublish':
                $this->deleteSyliusProduct($collector);
                break;
            case 'variant.unpublish':
                $this->deleteSyliusVariant($collector);
                break;
            case 'option.unpublish':
                $this->deleteSyliusOption($collector);
                break;
            default:
                throw new DeleteNotPossibleException('Cannot delete unknown object!');
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws DeleteNotPossibleException
     */
    private function deleteSyliusTaxon(DataCollectorInterface $collector)
    {
        try {
            $this->syliusTaxonHandler->delete($collector);
        } catch (RestRequestFailedException $exception) {
            throw new DeleteNotPossibleException('Cannot delete taxon.', 500, $exception);
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws DeleteNotPossibleException
     */
    private function deleteSyliusProduct(DataCollectorInterface $collector)
    {
        try {
            $this->syliusProductHandler->delete($collector);
        } catch (RestRequestFailedException $exception) {
            throw new DeleteNotPossibleException('Cannot delete product.', 500, $exception);
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws DeleteNotPossibleException
     */
    private function deleteSyliusVariant(DataCollectorInterface $collector)
    {
        try {
            $this->syliusVariantHandler->delete($collector);
        } catch (RestRequestFailedException $exception) {
            throw new DeleteNotPossibleException('Cannot delete variant.', 500, $exception);
        }
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws DeleteNotPossibleException
     */
    private function deleteSyliusOption(DataCollectorInterface $collector)
    {
        try {
            $this->syliusOptionHandler->delete($collector);
        } catch (RestRequestFailedException $exception) {
            throw new DeleteNotPossibleException('Cannot delete product option.', 500, $exception);
        }
    }
}
