<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\RabbitMQBundle\EventListener;

use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Pimcore\RabbitMQBundle\Event\PreUnpublishEvent;
use Lifestyle\Pimcore\RabbitMQBundle\EventListener\PreUnpublishListener;
use Lifestyle\Pimcore\RabbitMQBundle\Exception\DeleteNotPossibleException;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * Class PreUnpublishListenerTest
 * @package Tests\Lifestyle\Pimcore\RabbitMQBundle\EventListener
 */
class PreUnpublishListenerTest extends TestCase
{
    /**
     * @var HandlerInterface|MockObject
     */
    private $syliusTaxonHandlerMock;

    /**
     * @var HandlerInterface|MockObject
     */
    private $syliusProductHandlerMock;

    /**
     * @var HandlerInterface|MockObject
     */
    private $syliusVariantHandlerMock;

    /**
     * @var HandlerInterface|MockObject
     */
    private $syliusOptionHandlerMock;

    /**
     * @var PreUnpublishListener
     */
    protected $listener;

    protected function setUp()
    {
        $this->syliusTaxonHandlerMock = $this->getMockBuilder(HandlerInterface::class)->getMock();
        $this->syliusProductHandlerMock = $this->getMockBuilder(HandlerInterface::class)->getMock();
        $this->syliusVariantHandlerMock = $this->getMockBuilder(HandlerInterface::class)->getMock();
        $this->syliusOptionHandlerMock = $this->getMockBuilder(HandlerInterface::class)->getMock();

        $this->listener = new PreUnpublishListener(
            $this->syliusTaxonHandlerMock,
            $this->syliusProductHandlerMock,
            $this->syliusVariantHandlerMock,
            $this->syliusOptionHandlerMock
        );
    }

    public function testOnPreUnpublishWillThrowExceptionForUnknownEvent()
    {
        $this->expectException(DeleteNotPossibleException::class);
        $this->expectExceptionMessage('Cannot delete unknown object!');

        $collector = new DataCollector();
        $collector->addItem('event', 'UNKNOWN.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishWillDeleteTaxonSuccessfully()
    {
        $collector = new DataCollector();
        $collector->addItem('event', 'taxon.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusTaxonHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector);

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishDeleteTaxonFailedAndWillThrowException()
    {
        $this->expectException(DeleteNotPossibleException::class);
        $this->expectExceptionMessage('Cannot delete taxon.');
        $this->expectExceptionCode(500);

        $collector = new DataCollector();
        $collector->addItem('event', 'taxon.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusTaxonHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector)
            ->willThrowException(new RestRequestFailedException('unit_test_delete_failed'));

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishWillDeleteProductSuccessfully()
    {
        $collector = new DataCollector();
        $collector->addItem('event', 'product.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusProductHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector);

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishDeleteProductFailedAndWillThrowException()
    {
        $this->expectException(DeleteNotPossibleException::class);
        $this->expectExceptionMessage('Cannot delete product.');
        $this->expectExceptionCode(500);

        $collector = new DataCollector();
        $collector->addItem('event', 'product.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusProductHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector)
            ->willThrowException(new RestRequestFailedException('unit_test_delete_failed'));

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishWillDeleteVariantSuccessfully()
    {
        $collector = new DataCollector();
        $collector->addItem('event', 'variant.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusVariantHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector);

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishDeleteVariantFailedAndWillThrowException()
    {
        $this->expectException(DeleteNotPossibleException::class);
        $this->expectExceptionMessage('Cannot delete variant.');
        $this->expectExceptionCode(500);

        $collector = new DataCollector();
        $collector->addItem('event', 'variant.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusVariantHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector)
            ->willThrowException(new RestRequestFailedException('unit_test_delete_failed'));

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishWillDeleteOptionSuccessfully()
    {
        $collector = new DataCollector();
        $collector->addItem('event', 'option.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusOptionHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector);

        $this->listener->onPreUnpublish($event);
    }

    public function testOnPreUnpublishDeleteOptionFailedAndWillThrowException()
    {
        $this->expectException(DeleteNotPossibleException::class);
        $this->expectExceptionMessage('Cannot delete product option.');
        $this->expectExceptionCode(500);

        $collector = new DataCollector();
        $collector->addItem('event', 'option.unpublish');

        $event = new PreUnpublishEvent($collector);

        $this->syliusOptionHandlerMock->expects($this->once())
            ->method('delete')
            ->with($collector)
            ->willThrowException(new RestRequestFailedException('unit_test_delete_failed'));

        $this->listener->onPreUnpublish($event);
    }
}
