<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\RabbitMQBundle\Export;

use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Lifestyle\Pimcore\RabbitMQBundle\Exception\DeleteNotPossibleException;
use Lifestyle\Pimcore\RabbitMQBundle\Export\Config;
use Lifestyle\Pimcore\RabbitMQBundle\Export\Handler;
use Lifestyle\Pimcore\RabbitMQBundle\Producer\ProducerInterface;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Pimcore\RabbitMQBundle\Export
 */
class HandlerTest extends TestCase
{
    /**
     * @var SerializerInterface|MockObject
     */
    private $serializer;

    /**
     * @var LoggerInterface|MockObject
     */
    private $logger;

    /**
     * @var Config|MockObject
     */
    private $config;

    /**
     * @var ProducerInterface|MockObject
     */
    private $producer;

    /**
     * @var Handler
     */
    private $handler;

    /**
     * @var EventDispatcherInterface|MockObject
     */
    private $eventDispatcher;

    protected function setUp()
    {
        $this->serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $this->logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $this->config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $this->producer = $this->getMockBuilder(ProducerInterface::class)->getMock();
        $this->eventDispatcher = $this->getMockBuilder(EventDispatcherInterface::class)->getMock();

        $this->handler = new Handler(
            $this->serializer,
            $this->logger,
            $this->config,
            $this->producer,
            $this->eventDispatcher
        );
    }

    public function testIsResponsible()
    {
        $dataCollector = new DataCollector();
        $this->assertTrue($this->handler->isResponsible($dataCollector));
    }

    public function testPublishSuccessfully()
    {
        $this->logger->expects($this->once())
            ->method('info');

        $this->producer->expects($this->once())
            ->method('publish');

        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'UnitTest');

        $this->handler->publish($dataCollector);
    }

    public function testUnpublishSuccessfully()
    {
        $this->logger->expects($this->once())
            ->method('info');

        $this->producer->expects($this->once())
            ->method('publish');

        $this->eventDispatcher->expects($this->once())
            ->method('dispatch');

        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'UnitTest');

        $this->handler->unpublish($dataCollector);
    }

    public function testUnpublishNotPossibleBecauseSyliusDeleteRequestHasBeenFailed()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage("Unit Test Delete not Possible Exception");

        $this->logger->expects($this->never())
            ->method('info');

        $this->producer->expects($this->never())
            ->method('publish');

        $this->eventDispatcher->expects($this->once())
            ->method('dispatch')
            ->willThrowException(new DeleteNotPossibleException('Unit Test Delete not Possible Exception'));

        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'UnitTest');

        $this->handler->unpublish($dataCollector);
    }

    public function testSendFailed()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage("Could not send data to MessageQueue: Unit Test Error Message");

        $this->logger->expects($this->once())
            ->method('info');

        $this->producer->expects($this->once())
            ->method('publish')
            ->will($this->throwException(new \Exception('Unit Test Error Message')));

        $dataCollector = new DataCollector();
        $this->handler->publish($dataCollector);
    }
}
