<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\RabbitMQBundle\Export;

use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Lifestyle\Pimcore\RabbitMQBundle\Export\Config;
use PHPUnit\Framework\TestCase;

/**
 * Class ConfigTest
 * @package Tests\Lifestyle\Pimcore\RabbitMQBundle\Export
 */
class ConfigTest extends TestCase
{
    const EXPECTED_EXCEPTION_MESSAGE = 'Invalid or missing plugin configuration for RabbitMQ export handler!';

    public function testConfigIsInvalidNoVHost()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config(null, 'host', 8080, 'user', 'pass', 'exchange', 'json');
        $config->validate();
    }

    public function testConfigIsInvalidNoHost()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config('vhost', null, 8080, 'user', 'pass', 'exchange', 'json');
        $config->validate();
    }

    public function testConfigIsInvalidNoPort()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config('vhost', 'host', 0, 'user', 'pass', 'exchange', 'json');
        $config->validate();
    }

    public function testConfigIsInvalidNoUserName()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config('vhost', 'host', 8080, null, 'pass', 'exchange', 'json');
        $config->validate();
    }

    public function testConfigIsInvalidNoPassword()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config('vhost', 'host', 8080, 'user', null, 'exchange', 'json');
        $config->validate();
    }

    public function testConfigIsInvalidNoExchangeName()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config('vhost', 'host', 8080, 'user', 'pass', null, 'json');
        $config->validate();
    }

    public function testConfigIsInvalidInvalidRequestFormat()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage(self::EXPECTED_EXCEPTION_MESSAGE);

        $config = new Config('vhost', 'host', 8080, 'user', 'pass', 'exchange', 'invalid_format');
        $config->validate();
    }

    public function testConfigIsValid()
    {
        $config = new Config('vhost', 'host', 8080, 'user', 'pass', 'exchange', 'json');

        $this->assertSame('vhost', $config->getVHost());
        $this->assertSame('host', $config->getHost());
        $this->assertSame(8080, $config->getPort());
        $this->assertSame('user', $config->getUserName());
        $this->assertSame('pass', $config->getPassword());
        $this->assertSame('exchange', $config->getExchangeName());
        $this->assertSame('json', $config->getRequestFormat());

        try {
            $config->validate();
        } catch (\Exception $exception) {
            $this->fail('Config is valid but an exception was thrown!');
        }

        $this->assertTrue(true);
    }
}
