<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\RabbitMQBundle\Export;

use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Lifestyle\Pimcore\ExportBundle\Export\HandlerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\RabbitMQBundle\Event\PreUnpublishEvent;
use Lifestyle\Pimcore\RabbitMQBundle\Exception\DeleteNotPossibleException;
use Lifestyle\Pimcore\RabbitMQBundle\Producer\ProducerAdapter;
use Lifestyle\Pimcore\RabbitMQBundle\Producer\ProducerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\RabbitMQBundle\Export
 */
class Handler implements HandlerInterface
{
    const EVENT_PUBLISH_SUFFIX = '.publish';
    const EVENT_UNPUBLISH_SUFFIX = '.unpublish';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var ProducerInterface
     */
    private $producer;

    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param Config $config
     * @param ProducerInterface $producer
     * @param EventDispatcherInterface $dispatcher
     * @throws PimcoreExportFailedException
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        Config $config,
        $producer,
        EventDispatcherInterface $dispatcher
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->config = $config;
        $this->producer = $producer;
        $this->eventDispatcher = $dispatcher;

        $this->config->validate();
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector): bool
    {
        return !$collector->hasItem('className') || 0 !== strpos($collector->getItem('className')->getValue(), 'Asset');
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws PimcoreExportFailedException
     */
    public function publish(DataCollectorInterface $collector)
    {
        $this->send($collector);
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws PimcoreExportFailedException
     */
    public function unpublish(DataCollectorInterface $collector)
    {
        try {
            $event = new PreUnpublishEvent($collector);
            $this->eventDispatcher->dispatch(PreUnpublishEvent::NAME, $event);
        } catch (DeleteNotPossibleException $exception) {
            throw new PimcoreExportFailedException($exception->getMessage(), $exception->getCode(), $exception);
        }

        // Sylius DELETE request was successful, so we can send the unpublish request to the queue.
        // But Sylius Consumer shouldn't process the DELETE request again.
        $this->send($collector);
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws PimcoreExportFailedException
     */
    private function send($collector)
    {
        $this->logger->info(
            sprintf(
                'RabbitMQRequestHandler: sending request. host: "%s", port:"%s", exchange: "%s.',
                $this->config->getHost(),
                $this->config->getPort(),
                $this->config->getExchangeName()
            )
        );

        try {
            $this->producer->publish(
                $this->serializer->serialize($collector, $this->config->getRequestFormat()),
                '',
                [ProducerAdapter::PROPERTY_MESSAGE_DELIVERY_MODE => ProducerAdapter::MESSAGE_DELIVERY_MODE_PERSISTENCE]
            );
        } catch (\Exception $exception) {
            throw new PimcoreExportFailedException(
                sprintf('Could not send data to MessageQueue: %s', $exception->getMessage())
            );
        }
    }
}
