<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Lifestyle\Pimcore\Sso\Security\Authentication\Provider;

use Lifestyle\Pimcore\Sso\Exception\RuntimeException;
use Lifestyle\Pimcore\Sso\Security\Authentication\Token\SamlToken;
use Pimcore\Bundle\AdminBundle\Security\User\User;
use Pimcore\Bundle\AdminBundle\Security\User\UserProvider;
use Pimcore\Model\User as UserModel;
use Pimcore\Tool\Authentication;
use Pimcore\Tool\Session;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\Attribute\AttributeBagInterface;
use Symfony\Component\Security\Core\Role\Role;
use Symfony\Component\Security\Core\User\UserInterface;

/**
 * Class AdminUserProvider
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Pimcore\Sso\Security\Authentication\Provider
 */
class AdminUserProvider extends UserProvider implements SamlUserProviderInterface
{
    /** @var RequestStack */
    private $requestStack;

    /**
     * AdminUserProvider constructor.
     * @param RequestStack $requestStack
     */
    public function __construct(RequestStack $requestStack)
    {
        $this->requestStack = $requestStack;
    }

    /**
     * @param SamlToken $token
     * @return UserInterface
     * @throws \Exception
     */
    public function loadUserBySamlToken(SamlToken $token): UserInterface
    {
        $pimcoreUser = UserModel::getByName($token->getUsername());

        // Fallback to email address
        if (
            !$pimcoreUser instanceof UserModel &&
            $token->hasUserAttribute('email') &&
            $token->getUserAttribute('email')
        ) {
            $pimcoreUser = UserModel::getByName($token->getUserAttribute('email'));
        }

        if ($pimcoreUser instanceof UserModel) {
            $pimcoreUser = $this->updateUserBySamlToken($pimcoreUser, $token);
        } else {
            $pimcoreUser = $this->createUserBySamlToken($token);
        }

        // Pimcore wants the firewall configured with 'stateless=true' - handle session here
        if (Authentication::isValidUser($pimcoreUser)) {
            Session::useSession(function (AttributeBagInterface $adminSession) use ($pimcoreUser) {
                Session::regenerateId();
                $adminSession->set('user', $pimcoreUser);
            });
        }

        return new User($pimcoreUser);
    }

    /**
     * @param SamlToken $token
     * @return UserModel
     * @throws RuntimeException
     */
    private function createUserBySamlToken(SamlToken $token): UserModel
    {
        $username = $token->getUserAttribute('username');
        $plainPassword = substr(sha1(mt_rand() . $token->getUserAttribute('userIdentifier')), 0, 10);
        try {
            $password = Authentication::getPasswordHash($username, $plainPassword);
        } catch (\Exception $exception) {
            throw new RuntimeException('Unable to create generic password! ' . $exception->getMessage());
        }

        $pimcoreUser = UserModel::create([
            'type' => 'user',
            'name' => $username,
            'password' => $password,
            'firstname' => $token->getUserAttribute('firstName'),
            'lastname' => $token->getUserAttribute('lastName'),
            'email' => $token->getUserAttribute('email'),
            'language' => $this->getLocale($token),
            'lastLogin' => time(),
            'roles' => $this->getRoles($this->getRoleNames($token)),
            'admin' => $this->isAdmin($this->getRoleNames($token)),
            'active' => true,
        ]);

        if (!$pimcoreUser instanceof UserModel) {
            throw new RuntimeException(sprintf(
                'Unable to create pimcore user! Should be instance of "%s" but is "%s"',
                UserModel::class,
                gettype($pimcoreUser)
            ));
        }

        return $pimcoreUser;
    }

    /**
     * @param UserModel $pimcoreUser
     * @param SamlToken $token
     * @return UserModel
     * @throws \Exception
     */
    private function updateUserBySamlToken(UserModel $pimcoreUser, SamlToken $token): UserModel
    {
        $pimcoreUser->setFirstname($token->getUserAttribute('firstName'));
        $pimcoreUser->setLastname($token->getUserAttribute('lastName'));
        $pimcoreUser->setEmail($token->getUserAttribute('email'));
        $pimcoreUser->setLanguage($this->getLocale($token));
        $pimcoreUser->setLastLogin(time());
        $pimcoreUser->setRoles($this->getRoles($this->getRoleNames($token)));
        $pimcoreUser->setAdmin($this->isAdmin($this->getRoleNames($token)));
        $pimcoreUser->save();

        return $pimcoreUser;
    }

    /**
     * @param array $roleNames
     * @return bool
     */
    private function isAdmin(array $roleNames): bool
    {
        return in_array('ROLE_PIMCORE_ADMIN', $roleNames);
    }

    /**
     * @param array $roleNames
     * @return array
     */
    private function getRoles(array $roleNames): array
    {
        // No need to add this roles, because Pimcore does this depending on 'isAdmin'
        $roleNames = array_diff($roleNames, ['ROLE_PIMCORE_ADMIN', 'ROLE_PIMCORE_USER']);

        // Pimcore wants role-ids in its user object
        return array_filter(
            array_map(function (string $roleName) {
                $pimcoreRole = UserModel\Role::getByName($roleName);
                return null !== $pimcoreRole ? $pimcoreRole->getId() : null;
            }, $roleNames)
        );
    }

    /**
     * @param SamlToken $token
     * @return array
     */
    private function getRoleNames(SamlToken $token): array
    {
        return array_map(function(Role $role) {
            return $role->getRole();
        }, $token->getRoles());
    }

    /**
     * @param SamlToken $token
     * @return string
     */
    private function getLocale(SamlToken $token): string
    {
        if ($token->hasAttribute('locale')) {
            return $token->getAttribute('locale');
        }
        return $this->requestStack->getMasterRequest()->getLocale();
    }
}
