<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Lifestyle\Pimcore\Sso\Security\Factory;

use Lifestyle\Pimcore\Sso\EventListener\LoginRequestListener;
use Lifestyle\Pimcore\Sso\Security\Authentication\Provider\SamlProvider;
use Lifestyle\Pimcore\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Lifestyle\Pimcore\Sso\Security\Firewall\SamlListener;
use Lifestyle\Pimcore\Sso\Security\Http\EntryPoint\SamlAuthenticationEntryPoint;
use Symfony\Bundle\SecurityBundle\DependencyInjection\Security\Factory\SecurityFactoryInterface;
use Symfony\Component\Config\Definition\Builder\NodeDefinition;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Class SamlFactory
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Pimcore\Sso\Security\Factory
 */
class SamlFactory implements SecurityFactoryInterface
{
    /**
     * Defines the position at which the provider is called.
     * Possible values: pre_auth, form, http, and remember_me.
     *
     * @return string
     */
    public function getPosition()
    {
        return 'pre_auth';
    }

    /**
     * Defines the configuration key used to reference the provider
     * in the firewall configuration.
     *
     * @return string
     */
    public function getKey()
    {
        return 'sso_login';
    }

    /**
     * @inheritDoc
     */
    public function create(ContainerBuilder $container, $id, $config, $userProviderId, $defaultEntryPointId)
    {
        $authenticatorId = $this->createAuthenticator($container, $id, $config);
        $this->createLoginRequestListener($container, $id, $config, $authenticatorId);

        return [
            $this->createAuthProvider($container, $id, $config, $userProviderId),
            $this->createListener($container, $id, $config, $userProviderId, $authenticatorId),
            $this->createEntryPoint($container, $id, $config, $defaultEntryPointId, $authenticatorId),
        ];
    }

    /**
     * Define the firewalls configuration
     *
     * @param NodeDefinition $node
     */
    public function addConfiguration(NodeDefinition $node): void
    {
        $node
            ->children()
                ->scalarNode('application_name')->isRequired()->cannotBeEmpty()->end()
                ->scalarNode('service_provider_name')->isRequired()->cannotBeEmpty()->end()
                ->scalarNode('sso_context')->cannotBeEmpty()->defaultValue('admin')->end()
                ->scalarNode('check_path')->cannotBeEmpty()->defaultValue('lifestyle_sso_admin_user_login_check')->end()
                ->scalarNode('login_path')->cannotBeEmpty()->defaultValue('pimcore_admin_login')->end()
                ->scalarNode('default_target_path')->end()
            ->end();
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id The unique id of the firewall
     * @param array $config The options array for this listener
     * @param string $userProviderId The id of the user provider
     * @return string never null, the id of the authentication provider
     */
    protected function createAuthProvider(
        ContainerBuilder $container,
        string $id,
        array $config,
        string $userProviderId
    ): string {
        $providerId = 'lifestyle.pimcore.sso.security.authentication.provider.saml_provider.' . $id;
        $container
            ->setDefinition($providerId, new ChildDefinition(SamlProvider::class))
            ->addArgument(new Reference($userProviderId))
            ->addArgument($config['application_name']);

        return $providerId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $defaultEntryPoint
     * @param string $authenticatorId
     * @return string
     */
    protected function createEntryPoint(
        ContainerBuilder $container,
        string $id,
        array $config,
        ?string $defaultEntryPoint,
        string $authenticatorId
    ): string {
        $entryPointId = 'lifestyle.pimcore.sso.security.http.entry_point.saml_authentication_entry_point.' . $id;
        $container
            ->setDefinition($entryPointId, new ChildDefinition(SamlAuthenticationEntryPoint::class))
            ->addArgument(new Reference($authenticatorId));

        return $entryPointId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $userProvider
     * @param string $authenticatorId
     * @return string
     */
    protected function createListener(
        ContainerBuilder $container,
        string $id,
        array $config,
        string $userProvider,
        string $authenticatorId
    ): string {
        $listenerId = 'lifestyle.pimcore.sso.security.firewall.saml_listener.' . $id;
        $container
            ->setDefinition($listenerId, new ChildDefinition(SamlListener::class))
            ->addArgument(new Reference($authenticatorId))
            ->addArgument($config['check_path'])
            ->addArgument($config['application_name'])
            ->addArgument($id);

        return $listenerId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @return string
     */
    private function createAuthenticator(ContainerBuilder $container, string $id, array $config): string
    {
        $samlAuthenticatorId = 'lifestyle.pimcore.sso.security.authentication.simple_saml_authenticator.' . $id;
        $container
            ->setDefinition($samlAuthenticatorId, new ChildDefinition(SimpleSamlAuthenticator::class))
            ->setPublic(true)
            ->addArgument($config['check_path'])
            ->addArgument($config['default_target_path'])
            ->addArgument($config['service_provider_name'])
            ->addTag('lifestyle.pimcore.sso.security.authentication.simple_saml_authenticator.' . $config['sso_context']);

        // At this point we cannot change any existing service's arguments. But the logout handler needs
        // to know about its SAML authenticator to generate the redirect URL to the identity provider.
        // So each SAML authenticator is tagged with its context. Currently it's always 'admin'. But to be
        // prepared for an upcoming frontend login we use the same as in our Sylius SSO Plugin.

        return $samlAuthenticatorId;
    }

    /**
     * @param ContainerBuilder $container
     * @param string $id
     * @param array $config
     * @param string $authenticatorId
     * @return string
     */
    private function createLoginRequestListener(
        ContainerBuilder $container,
        string $id,
        array $config,
        string $authenticatorId
    ): string {
        $loginListenerId = 'lifestyle.pimcore.sso.event_listener.login_request_listener.' . $id;
        $container
            ->setDefinition($loginListenerId, new ChildDefinition(LoginRequestListener::class))
            ->addArgument(new Reference($authenticatorId))
            ->addArgument($config['login_path'])
            ->addTag('kernel.event_listener', ['event' => 'kernel.request', 'method' => 'handle']);

        return $loginListenerId;
    }
}
