<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Lifestyle\Pimcore\Sso\Security\Firewall;

use Lifestyle\Pimcore\Sso\Exception\RuntimeException;
use Lifestyle\Pimcore\Sso\Model\SamlResponseMapper;
use Lifestyle\Pimcore\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Lifestyle\Pimcore\Sso\Security\Authentication\Token\TokenFactory;
use Pimcore\Bundle\AdminBundle\Security\User\User;
use Pimcore\Cache\Runtime;
use Pimcore\Model\User as UserModel;
use Pimcore\Tool\Admin;
use Pimcore\Tool\Authentication;
use Pimcore\Tool\Session;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Attribute\AttributeBagInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\Security\Core\Authentication\AuthenticationManagerInterface;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authentication\Token\UsernamePasswordToken;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Firewall\LegacyListenerTrait;
use Symfony\Component\Security\Http\Firewall\ListenerInterface;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Logout\LogoutSuccessHandlerInterface;
use Symfony\Component\Translation\TranslatorInterface;

// For BC
if (!trait_exists(LegacyListenerTrait::class)) {
    require_once dirname(dirname(__DIR__)) . '/Event/LegacyListenerTrait.php';
}

/**
 * Class SamlListener
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Pimcore\Sso\Security\Firewall
 */
class SamlListener implements ListenerInterface, LoggerAwareInterface
{
    use LoggerAwareTrait;
    use LegacyListenerTrait;

    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * @var AuthenticationManagerInterface
     */
    private $authenticationManager;

    /**
     * @var HttpUtils
     */
    private $httpUtils;

    /**
     * @var TokenStorageInterface
     */
    private $tokenStorage;

    /**
     * @var SamlResponseMapper
     */
    private $samlResponseMapper;

    /**
     * @var TokenFactory
     */
    private $tokenFactory;

    /**
     * @var AuthenticationSuccessHandlerInterface
     */
    private $successHandler;

    /**
     * @var AuthenticationFailureHandlerInterface
     */
    private $failureHandler;

    /**
     * @var LogoutSuccessHandlerInterface
     */
    private $logoutHandler;

    /**
     * @var SimpleSamlAuthenticator
     */
    private $samlAuthenticator;

    /**
     * @var string
     */
    private $checkPath;

    /**
     * @var string
     */
    private $applicationName;

    /**
     * @var string
     */
    private $providerKey;

    /**
     * SamlListener constructor.
     * @param TranslatorInterface $translator
     * @param AuthenticationManagerInterface $authenticationManager
     * @param HttpUtils $httpUtils
     * @param TokenStorageInterface $tokenStorage
     * @param SamlResponseMapper $samlResponseMapper
     * @param TokenFactory $tokenFactory
     * @param AuthenticationSuccessHandlerInterface $successHandler
     * @param AuthenticationFailureHandlerInterface $failureHandler
     * @param LogoutSuccessHandlerInterface $logoutHandler
     * @param SimpleSamlAuthenticator $samlAuthenticator
     * @param string $checkPath
     * @param string $applicationName
     * @param string $providerKey
     */
    public function __construct(
        TranslatorInterface $translator,
        AuthenticationManagerInterface $authenticationManager,
        HttpUtils $httpUtils,
        TokenStorageInterface $tokenStorage,
        SamlResponseMapper $samlResponseMapper,
        TokenFactory $tokenFactory,
        AuthenticationSuccessHandlerInterface $successHandler,
        AuthenticationFailureHandlerInterface $failureHandler,
        LogoutSuccessHandlerInterface $logoutHandler,
        SimpleSamlAuthenticator $samlAuthenticator,
        string $checkPath,
        string $applicationName,
        string $providerKey
    ) {
        $this->translator = $translator;
        $this->authenticationManager = $authenticationManager;
        $this->httpUtils = $httpUtils;
        $this->tokenStorage = $tokenStorage;
        $this->samlResponseMapper = $samlResponseMapper;
        $this->tokenFactory = $tokenFactory;
        $this->successHandler = $successHandler;
        $this->failureHandler = $failureHandler;
        $this->logoutHandler = $logoutHandler;
        $this->samlAuthenticator = $samlAuthenticator;
        $this->checkPath = $checkPath;
        $this->applicationName = $applicationName;
        $this->providerKey = $providerKey;
    }

    /**
     * @param RequestEvent $event
     */
    public function __invoke(RequestEvent $event)
    {
        $request = $event->getRequest();

        if (!$request->hasSession()) {
            throw new RuntimeException('This authentication method requires a session.');
        }

        $response = null;
        if ($this->requiresAuthentication($request)) {
            // Login request (SAML response) - try to create or update Pimcore user
            try {
                if (null === $returnValue = $this->attemptAuthentication()) {
                    return;
                }
                if ($returnValue instanceof TokenInterface) {
                    $this->onAuthentication($request, $returnValue);
                    $response = $this->onSuccess($request, $returnValue);
                } elseif ($returnValue instanceof Response) {
                    $response = $returnValue;
                }
            } catch (AuthenticationException $exception) {
                $response = $this->onFailure($request, $exception);
            }
        } elseif ($pimcoreUser = Authentication::authenticateSession($request)) {
            if (!$this->samlAuthenticator->isAuthenticated()) {
                // SAML has already logged out
                $response = $this->onLogout($request);
            } else {
                // Not a login request - as we are flying stateless, we must handle the session
                $token = $this->restoreFromSession($pimcoreUser);
                $this->onAuthentication($request, $token);
                $this->tokenStorage->setToken($token);
            }
        }

        if (null !== $response) {
            $event->setResponse($response);
        }
    }

    /**
     * @param UserModel $pimcoreUser
     * @return TokenInterface
     */
    private function restoreFromSession(UserModel $pimcoreUser): TokenInterface
    {
        $user = new User($pimcoreUser);
        return $this->tokenFactory->createFromSession($user, $this->applicationName);
    }

    /**
     * @param TokenInterface $token
     */
    private function storeIntoSession(TokenInterface $token)
    {
        $pimcoreUser = $token->getUser()->getUser();
        if (Authentication::isValidUser($pimcoreUser)) {
            Session::useSession(function (AttributeBagInterface $adminSession) use ($pimcoreUser) {
                Session::regenerateId();
                $adminSession->set('user', $pimcoreUser);
            });
        }
    }

    /**
     * Returns true if login-check path has been reached
     *
     * @param Request $request
     * @return bool
     */
    private function requiresAuthentication(Request $request)
    {
        return $this->httpUtils->checkRequestPath($request, $this->checkPath);
    }

    /**
     * Process response from identity provider
     */
    private function attemptAuthentication()
    {
        if (!$this->samlAuthenticator->isAuthenticated()) {
            throw new AuthenticationException('User is not authenticated.');
        }

        // Create token from SAML response
        $samlResponse = $this->samlResponseMapper->mapResponse($this->applicationName, $this->samlAuthenticator);

        $failure = null;
        if (null === $samlResponse->getApplication()) {
            $failure = new AuthenticationException('User is not connected to this application.');
        }
        if (null === $samlResponse->getUser()) {
            $failure = new AuthenticationException('User has not been set in SAML response.');
        }

        $token = $this->tokenFactory->createFromResponse($this->applicationName, $samlResponse);

        if (null !== $failure) {
            $failure->setToken($token);
            throw $failure;
        }

        // Start authentication
        return $this->authenticationManager->authenticate($token);
    }

    /**
     * @param Request $request
     * @param TokenInterface $token
     * @throws \Exception
     */
    private function onAuthentication(Request $request, TokenInterface $token)
    {
        /** @var UserModel $user */
        $user = $token->getUser()->getUser();

        // set user language
        $request->setLocale($user->getLanguage());
        $this->translator->setLocale($user->getLanguage());

        // set user on runtime cache for legacy compatibility
        Runtime::set('pimcore_admin_user', $user);

        if ($user->isAdmin() && Admin::isMaintenanceModeScheduledForLogin()) {
            Admin::activateMaintenanceMode(Session::getSessionId());
            Admin::unscheduleMaintenanceModeOnLogin();
        }
    }

    /**
     * @param Request $request
     * @param TokenInterface $token
     * @return Response
     */
    private function onSuccess(Request $request, TokenInterface $token): Response
    {
        if (null !== $this->logger) {
            $this->logger->info('User has been authenticated successfully.', ['username' => $token->getUsername()]);
        }

        $this->tokenStorage->setToken($token);
        $this->storeIntoSession($token);

        // To keep it simple we call the success handler directly
        $response = $this->successHandler->onAuthenticationSuccess($request, $token);

        if (!$response instanceof Response) {
            throw new RuntimeException('Authentication Success Handler did not return a Response.');
        }

        return $response;
    }

    /**
     * @param Request $request
     * @param AuthenticationException $failed
     * @return Response
     */
    private function onFailure(Request $request, AuthenticationException $failed): Response
    {
        if (null !== $this->logger) {
            $this->logger->info('Authentication request failed.', ['exception' => $failed]);
        }

        $token = $this->tokenStorage->getToken();
        if ($token instanceof UsernamePasswordToken && $this->providerKey === $token->getProviderKey()) {
            $this->tokenStorage->setToken(null);
        }

        // To keep it simple we call the failure handler directly
        $response = $this->failureHandler->onAuthenticationFailure($request, $failed);

        if (!$response instanceof Response) {
            throw new RuntimeException('Authentication Failure Handler did not return a Response.');
        }

        return $response;
    }

    /**
     * @param Request $request
     * @return Response
     */
    private function onLogout(Request $request): Response
    {
        $response = $this->logoutHandler->onLogoutSuccess($request);

        if (!$response instanceof Response) {
            throw new RuntimeException('Logout Handler did not return a Response.');
        }

        return $response;
    }
}
