<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Lifestyle\Pimcore\Sso\Security\Logout;

use Lifestyle\Pimcore\Sso\Exception\InvalidConfigurationException;
use Lifestyle\Pimcore\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Pimcore\Bundle\AdminBundle\Security\LogoutSuccessHandler;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Http\Logout\LogoutSuccessHandlerInterface;

/**
 * Class AdminUserLogoutHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Pimcore\Sso\Security\Logout
 */
class AdminUserLogoutHandler implements LogoutSuccessHandlerInterface
{
    /**
     * @var SimpleSamlAuthenticator|null
     */
    private $samlAuthenticator;

    /**
     * @var LogoutSuccessHandler
     */
    private $pimcoreLogoutHandler;

    /**
     * AdminUserLogoutHandler constructor.
     * @param SimpleSamlAuthenticator|null $samlAuthenticator
     * @param LogoutSuccessHandler $pimcoreLogoutHandler
     */
    public function __construct(?SimpleSamlAuthenticator $samlAuthenticator, LogoutSuccessHandler $pimcoreLogoutHandler)
    {
        $this->samlAuthenticator = $samlAuthenticator;
        $this->pimcoreLogoutHandler = $pimcoreLogoutHandler;
    }

    /**
     * {@inheritdoc}
     */
    public function onLogoutSuccess(Request $request)
    {
        if (null === $this->samlAuthenticator) {
            throw new InvalidConfigurationException(
                'Unable to finish logout process! Invalid firewall configuration. Parameter "sso_context" must be set to "admin".'
            );
        }

        // Call Pimcore's default logout handler
        $response = $this->pimcoreLogoutHandler->onLogoutSuccess($request);
        $targetUrl = $response->getTargetUrl();

        // If SSO has not been logged out yet - trigger sso logout procedure
        if ($this->samlAuthenticator->isAuthenticated()) {
            $targetUrl = $this->samlAuthenticator->getLogoutURL($targetUrl);
        }

        return new RedirectResponse($targetUrl);
    }
}
