<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Lifestyle\Pimcore\Sso\Model;

use JMS\Serializer\Exception\Exception as SerializeException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\Sso\Security\Authentication\SimpleSamlAuthenticator;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class SamlResponseMapper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Lifestyle\Pimcore\Sso\Model
 */
class SamlResponseMapper implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Factory
     */
    private $factory;

    /**
     * SamlResponseMapper constructor.
     * @param SerializerInterface $serializer
     * @param Factory $factory
     */
    public function __construct(SerializerInterface $serializer, Factory $factory)
    {
        $this->serializer = $serializer;
        $this->factory = $factory;
    }

    /**
     * @param string $applicationName
     * @param SimpleSamlAuthenticator $samlAuthenticator
     * @return SsoResponse
     */
    public function mapResponse(string $applicationName, SimpleSamlAuthenticator $samlAuthenticator): SsoResponse
    {
        return $this->mapSamlAttributes(
            $samlAuthenticator->getAttributes(),
            $applicationName
        );
    }

    /**
     * @param array $samlAttributes
     * @param string $applicationName
     * @return SsoResponse
     */
    private function mapSamlAttributes(array $samlAttributes, string $applicationName): SsoResponse
    {
        $application = $this->mapSsoApplication($samlAttributes, $applicationName);
        $user = $this->mapSsoUser($samlAttributes);
        return $this->factory->ssoResponse($application, $user);
    }

    /**
     * @param array $samlAttributes
     * @param string $applicationName
     * @return SsoApplication|null
     */
    private function mapSsoApplication(array $samlAttributes, string $applicationName): ?SsoApplication
    {
        $key = 'application::' . $applicationName;
        if (!isset($samlAttributes[$key])) {
            return null;
        }

        $applicationData = is_array($samlAttributes[$key]) ? reset($samlAttributes[$key]) : $samlAttributes[$key];
        try {
            $application = $this->serializer->deserialize($applicationData, SsoApplication::class, 'json');
            return $application instanceof SsoApplication ? $application : null;
        } catch (SerializeException $exception) {
            $this->logger->warning('Invalid saml response data: ' . $exception->getMessage());
        }

        return null;
    }

    /**
     * @param array $samlAttributes
     * @return SsoUser
     */
    private function mapSsoUser(array $samlAttributes): SsoUser
    {
        return new SsoUser(
            $this->getSamlAttributeAsString($samlAttributes, 'username'),
            $this->getSamlAttributeAsString($samlAttributes, 'id'),
            $this->getSamlAttributeAsString($samlAttributes, 'guid'),
            $this->getSamlAttributeAsString($samlAttributes, 'useridentifier'),
            $this->getSamlAttributeAsString($samlAttributes, 'firstname'),
            $this->getSamlAttributeAsString($samlAttributes, 'lastname'),
            $this->getSamlAttributeAsString($samlAttributes, 'email')
        );
    }

    /**
     * @param array $samlAttributes
     * @param string $attributeName
     * @return string|null
     */
    private function getSamlAttributeAsString(array $samlAttributes, string $attributeName): ?string
    {
        if (!isset($samlAttributes[$attributeName])) {
            return null;
        }

        $attribute = $samlAttributes[$attributeName];
        return (string)(is_array($attribute) ? reset($attribute) : $attribute);
    }
}
