<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SwooxBundle\Export;

use GuzzleHttp\ClientInterface;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Export\HandlerInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\SwooxBundle\Export
 */
class Handler implements HandlerInterface
{
    const HTTP_METHOD_PUT = 'PUT';
    const HTTP_METHOD_DELETE = 'DELETE';

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var Config
     */
    private $config;

    /**
     * Handler constructor.
     * @param ClientInterface $client
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param Mapper $mapper
     * @param Config $config
     */
    public function __construct(
        ClientInterface $client,
        SerializerInterface $serializer,
        LoggerInterface $logger,
        Mapper $mapper,
        Config $config
    ) {
        $this->client = $client;
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->mapper = $mapper;
        $this->config = $config;

        $this->config->validate();
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return true;
    }

    /**
     * @param DataCollectorInterface $collector
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        $request = $this->mapper->mapUpdateRequestModel($collector);
        $this->send(self::HTTP_METHOD_PUT, $this->config->getIndexUrl(), $request);
    }

    /**
     * @param DataCollectorInterface $collector
     */
    public function delete(DataCollectorInterface $collector)
    {
        $request = $this->mapper->mapDeleteRequestModel($collector);
        $this->send(self::HTTP_METHOD_DELETE, $this->config->getDeleteUrl(), $request);
    }

    /**
     * @param string $method
     * @param string $url
     * @param object $request
     * @throws PimcoreExportFailedException
     */
    private function send($method, $url, $request)
    {
        // Prepare request
        $options = array(
            'headers' => [
                'clientid' => $this->config->getClientId(),
            ],
            'body' => $this->serializer->serialize($request, $this->config->getRequestFormat())
        );

        $this->logger->info(sprintf('SwooxRequestHandler: sending request for client "%s" url %s', $this->config->getClientId(), $url));
        $this->logger->debug(print_r($options, true));

        // Send request
        try {
            $webserviceResponse = $this->client->request($method, $url, $options);

            $this->logger->info('SwooxRequestHandler: received response.');
            $this->logger->debug($webserviceResponse->getBody()->getContents());
        } catch (\Exception $exception) {
            // Received an error response
            throw new PimcoreExportFailedException($exception->getMessage());
        }
    }
}
