<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\Sylius\Export\Mapping\Field;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\DataCollector\DataCollectorItem;
use Lifestyle\Pimcore\ExportBundle\Mapping\Field\FieldMapperInterface;
use Pimcore\Model\DataObject\ClassDefinition\Data;
use Pimcore\Model\DataObject\ClassDefinition\Data\Classificationstore;
use Pimcore\Model\DataObject\Classificationstore as ClassificationstoreModel;
use Pimcore\Model\DataObject\Classificationstore\KeyConfig;
use Pimcore\Model\DataObject\Concrete;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Class ClassificationStoreMapper
 *
 * Replaces the default mapper Lifestyle\Pimcore\ExportBundle\Mapping\Field\ClassificationStoreMapper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\Sylius\Export\Mapping\Field
 */
class ClassificationStoreMapper implements FieldMapperInterface
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * ClassificationStoreMapper constructor.
     * @param TranslatorInterface $translator
     */
    public function __construct(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    /**
     * @param Data $property
     * @return bool
     */
    public function handles(Data $property)
    {
        return $property instanceof Classificationstore;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Concrete               $object
     * @param Data                   $property
     * @param string                 $language
     */
    public function handle(DataCollectorInterface $collector, Concrete $object, Data $property, $language)
    {
        $propertyName = $property->getName();
        $groups = $this->getGroups($object, $propertyName);

        // Get keys even if no value has been set - Sylius product needs product options to be prepared for variants
        $translatedAttributeKeys = $this->getDefinitions($object, $groups, $propertyName, $language);

        $attributeValues = [];
        $translatedAttributeValues = [];
        $attributeKeys = array_combine(array_keys($translatedAttributeKeys), array_keys($translatedAttributeKeys));
        foreach ($this->getClassificationStoreItems($object, $groups, $propertyName, $language) as $group) {
            foreach ($group as $keyId => $value) {
                $keyConfig = KeyConfig::getById($keyId);
                $key = $keyConfig->getName();
                $type = $keyConfig->getType();

                if (is_array($value)) {
                    $translatedValue = array_map(function ($value) use ($language, $type) {
                        $translatedRawValue = $this->translator->trans($value, [], null, $language);

                        return $this->isWysiwyg($type) ? $translatedRawValue : trim(strip_tags($translatedRawValue));
                    }, $value);
                } else {
                    $translatedRawValue = $this->translator->trans($value, [], null, $language);
                    $translatedValue =
                        $this->isWysiwyg($type) ? $translatedRawValue : trim(strip_tags($translatedRawValue));
                }

                $attributeValues[$propertyName . '_' . $key] = $value;
                $translatedAttributeValues[$propertyName . '_translated_' . $key] = $translatedValue;
                $attributeKeys[$key] = $key;
            }
        }

        // Add list of attribute keys: $propertyName => $attributeKeys
        $collector->replaceItem(new DataCollectorItem($propertyName, array_values($attributeKeys)));

        // Add list of translated attribute keys: $propertyName . '_translated' => translated-key
        $collector->replaceItem(new DataCollectorItem(
            $propertyName . '_translated',
            $translatedAttributeKeys,
            $language
        ));

        // Create one swoox slot for each attribute: $propertyName . '_' . $key => $attributeValue
        foreach ($attributeValues as $key => $value) {
            $collector->replaceItem(new DataCollectorItem($key, $value));
        }

        // Create one swoox slot for each translated attribute: $propertyName . '_translated_' . $key => $attributeValue
        foreach ($translatedAttributeValues as $key => $value) {
            $collector->addItem($key, $value, $language);
        }
    }

    /**
     * @param Concrete $object
     * @param array    $groups
     * @param string   $propertyName
     * @param string   $language
     * @return array
     */
    private function getClassificationStoreItems(
        Concrete $object,
        array $groups,
        string $propertyName,
        string $language
    ): array {
        $items = [];

        foreach ($groups as $groupId => $groupConfig) {
            foreach ($groupConfig->getRelations() as $relation) {
                /** @var \Pimcore\Model\DataObject\Classificationstore\KeyGroupRelation $relation */
                $items[$groupId][$relation->getKeyId()] = null;
            }
        }

        /** @var ClassificationstoreModel $classificationStore */
        $classificationStore = $object->getValueForFieldName($propertyName);

        // Convert item values
        foreach ($classificationStore->getItems() as $groupId => $group) {
            foreach (array_keys($group) as $keyId) {
                if (
                    array_key_exists($groupId, $items) &&
                    array_key_exists($keyId, $items[$groupId])
                ) {
                    $items[$groupId][$keyId] = $classificationStore->getLocalizedKeyValue($groupId, $keyId, $language);
                }
            }
        }

        $parent = $object->getNextParentForInheritance();
        if (null !== $parent) {
            $parentClassificationStore = $parent->getValueForFieldName($propertyName);

            // Add items from product that are not set in product-variant
            foreach ($parentClassificationStore->getItems() as $groupId => $group) {
                foreach (array_keys($group) as $keyId) {
                    if (
                        array_key_exists($groupId, $items) &&
                        array_key_exists($keyId, $items[$groupId]) &&
                        null === $items[$groupId][$keyId]
                    ) {
                        $items[$groupId][$keyId] = $parentClassificationStore->getLocalizedKeyValue(
                            $groupId,
                            $keyId,
                            $language
                        );
                    }
                }
            }
        }

        return $items;
    }

    /**
     * @param Concrete $object
     * @param array    $groups
     * @param string   $propertyName
     * @param string   $language
     * @return array
     */
    private function getDefinitions(Concrete $object, array $groups, string $propertyName, string $language): array
    {
        $keys = [];

        foreach ($groups as $groupId => $groupConfig) {
            foreach ($groupConfig->getRelations() as $relation) {
                /** @var \Pimcore\Model\DataObject\Classificationstore\KeyGroupRelation $relation */
                $keys[$relation->getName()] = trim(strip_tags($this->translator->trans(
                    $relation->getName(),
                    [],
                    null,
                    $language
                )));
            }
        }

        return $keys;
    }

    /**
     * @param Concrete $object
     * @param string   $propertyName
     * @return array
     */
    private function getGroups(Concrete $object, string $propertyName): array
    {
        $parent = $object->getNextParentForInheritance();
        $baseObject = null !== $parent ? $parent : $object;

        /** @var ClassificationstoreModel $classificationStore */
        $classificationStore = $baseObject->getValueForFieldName($propertyName);

        $groups = [];

        foreach (array_keys($classificationStore->getActiveGroups()) as $groupId) {
            $groups[$groupId] = \Pimcore\Model\DataObject\Classificationstore\GroupConfig::getById($groupId);
        }

        return $groups;
    }

    /**
     * @param string $fieldType
     * @return bool
     */
    private function isWysiwyg(string $fieldType): bool
    {
        // Textarea might contain html-markup
        return in_array(strtolower($fieldType), ['textarea', 'wysiwyg']);
    }
}
