<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SyliusBundle\Export\Taxon\Version1\Update;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\SyliusBundle\Export\Taxon\Version1\Update
 */
class Handler
{
    const API_UPDATE_URI = '/api/v1/taxons/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Config $config
     * @param Mapper $mapper
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Config $config,
        Mapper $mapper
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->config = $config;
        $this->mapper = $mapper;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws PimcoreExportFailedException
     */
    public function update(DataCollectorInterface $collector)
    {
        if (!strlen($collector->getItemValue('id'))) {
            throw new PimcoreExportFailedException('An unexpected error occurred while updating the object: Taxon code is missing.');
        }

        $taxon = $this->mapper->mapRequestModel($collector);

        $options = [
            'body' => $this->serializer->serialize($taxon, 'json')
        ];

        $uri = $this->buildUri($collector->getItemValue('id'));

        $this->logger->info(sprintf('Send PUT request %s with body: %s', $uri, $options['body']));

        try {
            $response = $this->client->request('PUT', $uri, $options);
        } catch (\Exception $exception) {
            throw new PimcoreExportFailedException('Could not update taxon object: ' . $exception->getMessage());
        }

        return 204 === $response->getStatusCode();
    }

    /**
     * @param string $code
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_UPDATE_URI, $code);
    }
}
