<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SyliusBundle\Export\Taxon\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\SyliusBundle\Export\Taxon\Version1\Get
 */
class Handler
{
    const API_GET_URI = '/api/v1/taxons/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Config
     */
    private $config;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Config $config
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Config $config
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->config = $config;
    }

    /**
     * @param string $code
     * @return bool
     * @throws PimcoreExportFailedException
     */
    public function exists($code)
    {
        try {
            $response = $this->getByCode($code);
        } catch (ClientException $exception) {
            if (404 === $exception->getResponse()->getStatusCode()) {
                return false;
            }

            throw new PimcoreExportFailedException('An unexpected error occurred while saving the object: ' . $exception->getMessage());
        } catch (\Exception $exception) {
            throw new PimcoreExportFailedException('An unexpected error occurred while saving the object: ' . $exception->getMessage());
        }

        return 200 === $response->getStatusCode();
    }

    /**
     * @param string $code
     * @return \Psr\Http\Message\ResponseInterface
     * @throws PimcoreExportFailedException
     */
    private function getByCode($code)
    {
        if (!strlen($code)) {
            throw new PimcoreExportFailedException('An unexpected error occurred while saving the object: Taxon code is missing.');
        }

        $uri = $this->buildUri($code);

        $this->logger->info(sprintf('Send GET request %s', $uri));

        return $this->client->request('GET', $uri);
    }

    /**
     * @param string $code
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_GET_URI, $code);
    }
}
