<?php

/**
 * Class OAuthMiddlewareAdapter
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SyliusBundle\Helper\Guzzle;

use Lifestyle\Pimcore\SyliusBundle\Persistence\PersistenceInterface;
use GuzzleHttp\ClientInterface;
use Sainsburys\Guzzle\Oauth2\AccessToken;
use Sainsburys\Guzzle\Oauth2\GrantType\GrantTypeInterface;
use Sainsburys\Guzzle\Oauth2\GrantType\RefreshTokenGrantTypeInterface;
use Sainsburys\Guzzle\Oauth2\Middleware\OAuthMiddleware;

/**
 * Class OAuthMiddlewareAdapter
 * @package Lifestyle\Pimcore\SyliusBundle\Helper\Guzzle
 */
class OAuthMiddlewareAdapter extends OAuthMiddleware
{
    const PERSISTENCE_KEY = 'sylius_oauth2_access_token';

    /**
     * @var PersistenceInterface
     */
    private $persistenceLayer;

    /**
     * OAuthMiddlewareAdapter constructor.
     * @param PersistenceInterface $persistenceLayer
     * @param ClientInterface $client
     * @param GrantTypeInterface|null $grantType
     * @param RefreshTokenGrantTypeInterface|null $refreshTokenGrantType
     */
    public function __construct(
        PersistenceInterface $persistenceLayer,
        ClientInterface $client,
        GrantTypeInterface $grantType = null,
        RefreshTokenGrantTypeInterface $refreshTokenGrantType = null
    ) {
        $this->persistenceLayer = $persistenceLayer;

        parent::__construct($client, $grantType, $refreshTokenGrantType);
    }

    public function getAccessToken()
    {
        // First try to load access token from persistence layer
        if (!$this->hasAccessToken()) {
            $this->loadAccessToken();
        }

        // Afterwards, if no valid access token could be loaded acquire a new one and persist it.
        if (!$this->hasAccessToken() || $this->accessToken->isExpired()) {
            $this->acquireAccessToken();
            $this->persistAccessToken();
        }

        return $this->accessToken;
    }

    /**
     * @return bool
     */
    public function hasAccessToken()
    {
        return $this->accessToken instanceof AccessToken;
    }

    private function loadAccessToken()
    {
        $accessTokenData = $this->persistenceLayer->load(self::PERSISTENCE_KEY);

        if ($accessTokenData['token'] && $accessTokenData['expires']) {
            $accessToken = new AccessToken($accessTokenData['token'], $accessTokenData['type'], ['expires' => $accessTokenData['expires']]);
            if (!$accessToken->isExpired()) {
                $this->setAccessToken($accessToken);
            }
        }
    }

    private function persistAccessToken()
    {
        $this->persistenceLayer->persist(self::PERSISTENCE_KEY, [
            'token' => $this->accessToken->getToken(),
            'type' => $this->accessToken->getType(),
            'expires' => $this->accessToken->getExpires()->getTimestamp(),
        ]);
    }
}
