<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Get\Model\Response\ProductVariant;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Get
 */
class Handler
{
    const API_GET_URI = '/api/v1/products/%s/variants/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Config
     */
    private $config;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Config $config
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Config $config
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->config = $config;
    }

    /**
     * @param string $parentCode
     * @param string $variantCode
     * @return bool
     * @throws PimcoreExportFailedException
     */
    public function exists($parentCode, $variantCode)
    {
        return false !== $this->getByCode($parentCode, $variantCode);
    }

    /**
     * @param string $parentCode
     * @param string $variantCode
     * @return ProductVariant|false, if no entity has been found
     * @throws PimcoreExportFailedException
     */
    public function getByCode($parentCode, $variantCode)
    {
        if (!strlen($parentCode) || !strlen($variantCode)) {
            throw new PimcoreExportFailedException('An unexpected error occurred while fetching the object: Product or variant code is missing.');
        }

        $uri = $this->buildUri($parentCode, $variantCode);

        $this->logger->info(sprintf('Send GET request %s', $uri));

        try {
            $response = $this->client->request('GET', $uri);
        } catch (ClientException $exception) {
            if (404 === $exception->getResponse()->getStatusCode()) {
                return false;
            }

            throw new PimcoreExportFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage())
            );
        } catch (\Exception $exception) {
            throw new PimcoreExportFailedException(
                sprintf('An unexpected error occurred while fetching the object: %s', $exception->getMessage())
            );
        }

        return 200 === $response->getStatusCode() ? $this->getResponseModel($response) : false;
    }

    /**
     * @param ResponseInterface $response
     * @return ProductVariant
     */
    private function getResponseModel(ResponseInterface $response)
    {
        return $this->serializer->deserialize($response->getBody()->getContents(), ProductVariant::class, 'json');
    }

    /**
     * @param string $parentCode
     * @param string $variantCode
     * @return string
     */
    private function buildUri($parentCode, $variantCode)
    {
        return sprintf(self::API_GET_URI, $parentCode, $variantCode);
    }
}
