<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Create;

use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollector;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Create\Handler;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Create\Mapper;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Create
 */
class HandlerTest extends TestCase
{
    public function testCreateSuccess()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(201, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');
        $collector->addItem('id', '123456');

        $createHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $this->assertTrue($createHandler->create($collector));
    }

    public function testCreateRequestUnexpectedClientException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('Could not add product variant object: Mocked Client Exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("Mocked Client Exception", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');
        $collector->addItem('id', '123456');

        $createHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $createHandler->create($collector);
    }

    public function testCreateRequestUnexpectedException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('Could not add product variant object: unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('parentId', '123456');
        $collector->addItem('id', '123456');

        $createHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $createHandler->create($collector);
    }

    public function testCreateRequestMissingParentIdException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while saving the object: Parent product code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $createHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $createHandler->create($collector);
    }
}
