<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1;

use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollector;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Create\Handler as CreateHandler;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Delete\Handler as DeleteHandler;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Get\Handler as GetHandler;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Get\Model\Response\ProductVariant;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Handler;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Update\Handler as UpdateHandler;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Factory;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1
 */
class HandlerTest extends TestCase
{
    /**
     * @var Handler
     */
    private $productHandler;

    /**
     * @var CreateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $createHandler;

    /**
     * @var DeleteHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $deleteHandler;

    /**
     * @var GetHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $getHandler;

    /**
     * @var UpdateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $updateHandler;

    protected function setUp()
    {
        parent::setUp();

        $this->createHandler = $this->getMockBuilder(CreateHandler::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->deleteHandler = $this->getMockBuilder(DeleteHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['deleteByCode'])
            ->getMock();
        $this->getHandler = $this->getMockBuilder(GetHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['exists', 'getByCode'])
            ->getMock();
        $this->updateHandler = $this->getMockBuilder(UpdateHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['update'])
            ->getMock();

        $serviceFactory = new Factory($this->createHandler, $this->deleteHandler, $this->getHandler, $this->updateHandler);
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        $this->productHandler = new Handler($serviceFactory, $logger);
    }

    public function testIsResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'Variant');

        $this->assertTrue($this->productHandler->isResponsible($dataCollector));
    }

    public function testIsNotResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'SomeOtherClass');

        $this->assertFalse($this->productHandler->isResponsible($dataCollector));
    }

    public function testCreate()
    {
        $this->getHandler->expects($this->once())
            ->method('getByCode')
            ->willReturn(false);

        $this->createHandler->expects($this->once())
            ->method('create');

        $dataCollector = new DataCollector();

        $this->productHandler->createOrUpdate($dataCollector);
    }

    public function testUpdateAndApplyVersion()
    {
        $responseModel = new ProductVariant(
            123, "test", 0, 10, 0, 10, true
        );

        $this->getHandler->expects($this->once())
            ->method('getByCode')
            ->willReturn($responseModel);

        $this->updateHandler->expects($this->once())
            ->method('update');

        $dataCollector = new DataCollector();
        $this->productHandler->createOrUpdate($dataCollector);

        // Make sure the corresponding version is added.
        $this->assertSame(10, $dataCollector->getItemValue('version'));
    }

    public function testDelete()
    {
        $productCode = 'unitTest123';
        $variantCode = 'variantCode123';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($productCode, $variantCode)
            ->willReturn(true);

        $this->deleteHandler->expects($this->once())
            ->method('deleteByCode')
            ->with($productCode, $variantCode);

        $dataCollector = new DataCollector();
        $dataCollector->addItem('parentId', $productCode);
        $dataCollector->addItem('id', $variantCode);

        $this->productHandler->delete($dataCollector);
    }

    public function testDoNotDelete()
    {
        $productCode = 'unitTest123';
        $variantCode = 'variantCode123';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($productCode, $variantCode)
            ->willReturn(false);

        $this->deleteHandler->expects($this->never())
            ->method('deleteByCode');

        $dataCollector = new DataCollector();
        $dataCollector->addItem('parentId', $productCode);
        $dataCollector->addItem('id', $variantCode);

        $this->productHandler->delete($dataCollector);
    }
}
