<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Create;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Create
 */
class Handler
{
    const API_CREATE_URI = '/api/v1/products/';
    const CLIENT_EXCEPTION_MESSAGE = 'Could not add product object!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Config $config
     * @param Mapper $mapper
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Config $config,
        Mapper $mapper
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->config = $config;
        $this->mapper = $mapper;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws PimcoreExportFailedException
     */
    public function create(DataCollectorInterface $collector)
    {
        $product = $this->mapper->mapRequestModel($collector);

        $options = [
            'body' => $this->serializer->serialize($product, 'json')
        ];

        $this->logger->info(
            sprintf('Send POST request to %s with body: %s', self::API_CREATE_URI, $options['body'])
        );

        try {
            $response = $this->client->request('POST', self::API_CREATE_URI, $options);
        } catch (ClientException $exception) {
            throw new PimcoreExportFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents()
            );
        } catch (\Exception $exception) {
            throw new PimcoreExportFailedException(self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage());
        }

        return 201 === $response->getStatusCode();
    }
}
