<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Delete;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Delete
 */
class Handler
{
    const API_DELETE_URI = '/api/v1/products/%s';
    const CLIENT_EXCEPTION_MESSAGE = 'Deleting the product has been failed!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Config
     */
    private $config;

    /**
     * Handler constructor.
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Config $config
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Config $config
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->config = $config;
    }

    /**
     * @param string $code
     * @return bool
     * @throws PimcoreExportFailedException
     */
    public function deleteByCode($code)
    {
        if (!strlen($code)) {
            throw new PimcoreExportFailedException(
                'An unexpected error occurred while deleting the object: Product code is missing.'
            );
        }

        $uri = $this->buildUri($code);

        $this->logger->info(sprintf('Send DELETE request %s', $uri));

        try {
            $response = $this->client->request('DELETE', $uri);
        } catch (ClientException $exception) {
            throw new PimcoreExportFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents()
            );
        } catch (\Exception $exception) {
            throw new PimcoreExportFailedException(self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage());
        }

        return 204 === $response->getStatusCode();
    }

    /**
     * @param string $code
     * @return string
     */
    private function buildUri($code)
    {
        return sprintf(self::API_DELETE_URI, $code);
    }
}
