<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Update;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\ExportBundle\Mapping\DataCollector\DataCollector;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Update\Handler;
use Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Update\Mapper;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Pimcore\SyliusBundle\Export\Product\Version1\Update
 */
class HandlerTest extends TestCase
{
    public function testUpdateSuccess()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(204, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $this->assertTrue($updateHandler->update($collector));
    }

    public function testUpdateRequestUnexpectedClientException()
    {
        $this->expectException(PimcoreExportFailedException::class);

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $updateHandler->update($collector);
    }

    public function testUpdateRequestUnexpectedException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('Could not update product object! - unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();
        $collector->addItem('id', '123456');

        $updateHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $updateHandler->update($collector);
    }

    public function testUpdateNoCodeException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while updating the object: Product code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $collector = new DataCollector();

        $updateHandler = new Handler($serializer, $logger, $client, $config, $mapper);
        $updateHandler->update($collector);
    }

    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();

        $updateHandler = new Handler($serializer, $logger, $client, $config, $mapper);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($updateHandler, 'productCode12345');
        $this->assertSame('/api/v1/products/productCode12345', $uri);
    }
}
