<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Delete;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Pimcore\SyliusBundle\Export\Config;
use Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Delete\Handler;
use Lifestyle\Pimcore\ExportBundle\Exception\PimcoreExportFailedException;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Pimcore\SyliusBundle\Export\ProductVariant\Version1\Delete
 */
class HandlerTest extends TestCase
{
    public function testDeleteByCodeSuccess()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(204, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client, $config);
        $this->assertTrue($deleteHandler->deleteByCode('unitTestParent123', 'unitTestVariant1234'));
    }

    public function testDeleteRequestUnexpectedClientException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('Deleting the product variant has been failed! - ');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("Mocked Client Exception", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client, $config);
        $deleteHandler->deleteByCode('unitTestParent123', 'unitTestVariant1234');
    }

    public function testDeleteRequestUnexpectedException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('Deleting the product variant has been failed! - unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client, $config);
        $deleteHandler->deleteByCode('unitTestParent123', 'unitTestVariant1234');
    }

    public function testDeleteByCodeThrowsNoParentCodeException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while deleting the object: Product code or variant code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client, $config);
        $deleteHandler->deleteByCode(null, 'unitTestVariant1234');
    }

    public function testDeleteByCodeThrowsNoVariantCodeException()
    {
        $this->expectException(PimcoreExportFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while deleting the object: Product code or variant code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client, $config);
        $deleteHandler->deleteByCode('unitTestVariant1234', null);
    }

    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();

        $deleteHandler = new Handler($serializer, $logger, $client, $config);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($deleteHandler, 'productParentCode12345', 'productVariantCode12345');
        $this->assertSame('/api/v1/products/productParentCode12345/variants/productVariantCode12345', $uri);
    }
}
