<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\Taxon\Validator\Constraints;

use Lifestyle\Pimcore\Taxon\Configuration\TaxonConfiguration;
use Pimcore\Model\DataObject;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Class TaxonValidator
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\Taxon\Validator\Constraints
 */
class TaxonValidator extends ConstraintValidator
{
    /**
     * @var TaxonConfiguration
     */
    private $productConfiguration;

    /**
     * TaxonValidator constructor.
     * @param TaxonConfiguration $productConfiguration
     */
    public function __construct(TaxonConfiguration $productConfiguration)
    {
        $this->productConfiguration = $productConfiguration;
    }

    /**
     * Checks if the passed value is valid.
     *
     * @param mixed $value The value that should be validated
     * @param Constraint $constraint The constraint for the validation
     */
    public function validate($value, Constraint $constraint)
    {
        if (null === $value || '' === $value) {
            return;
        }

        if (!is_int($value)) {
            throw new UnexpectedTypeException($value, 'int');
        }

        $object = DataObject::getById($value);
        if (null === $object) {
            $this->context->buildViolation($constraint->objectNotFoundMessage)
                ->setParameter('{{ id }}', $value)
                ->addViolation();
        } else {
            $mandatoryProperties = [
                $this->productConfiguration->getUrlPropertyName(),
            ];
            foreach ($mandatoryProperties as $property) {
                if (!$this->validateObjectProperty($object, $property)) {
                    $this->context->buildViolation($constraint->invalidPropertyMessage)
                        ->setParameter('{{ id }}', $value)
                        ->setParameter('{{ className }}', get_class($object))
                        ->setParameter('{{ property }}', $property)
                        ->addViolation();
                }
            }
        }
    }

    /**
     * @param DataObject $object
     * @param string $property
     * @return bool
     */
    private function validateObjectProperty(DataObject $object, string $property): bool
    {
        $reflectionClass = new \ReflectionClass($object);
        $getter = 'get' . ucfirst($property);
        $setter = 'set' . ucfirst($property);
        return $reflectionClass->hasMethod($getter) && $reflectionClass->hasMethod($setter);
    }
}
