<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\Taxon\MessageHandler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\Taxon\Configuration\TaxonConfiguration;
use Lifestyle\Pimcore\Taxon\Exception\ExceptionInterface;
use Lifestyle\Pimcore\Taxon\Exception\InvalidArgumentException;
use Lifestyle\Pimcore\Taxon\Exception\NotFoundException;
use Lifestyle\Pimcore\Taxon\Model\Taxon;
use Lifestyle\Pimcore\Taxon\Model\TaxonMapper;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Pimcore\Model\DataObject;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\ConstraintViolationInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Component\Validator\ConstraintViolationListInterface;

/**
 * Class TaxonHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\Taxon\MessageHandler
 */
class TaxonHandler implements HandlerInterface
{
    /**
     * @var TaxonConfiguration
     */
    private $taxonConfiguration;

    /**
     * @var TaxonMapper
     */
    private $mapper;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * TaxonHandler constructor.
     * @param TaxonConfiguration $taxonConfiguration
     * @param TaxonMapper $mapper
     * @param ValidatorInterface $validator
     */
    public function __construct(
        TaxonConfiguration $taxonConfiguration,
        TaxonMapper $mapper,
        ValidatorInterface $validator
    ) {
        $this->taxonConfiguration = $taxonConfiguration;
        $this->mapper = $mapper;
        $this->validator = $validator;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws ExceptionInterface
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $taxonData = $this->mapper->mapTaxon($dataCollector);
        $violations = $this->validator->validate($taxonData);
        if (0 < count($violations)) {
            throw new InvalidArgumentException(
                $this->violationsToString($violations),
                Response::HTTP_BAD_REQUEST
            );
        }

        $taxon = DataObject::getById($taxonData->getId(), true);
        if (null === $taxon) {
            throw new NotFoundException(
                sprintf('Object (%d) not found.', $taxonData->getId()),
                Response::HTTP_NOT_FOUND
            );
        }

        return $this->execute($taxon, $taxonData);
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            $dataCollector->hasItem('event') &&
            $dataCollector->getItemValue('event') === 'taxon.url.update';
    }

    /**
     * @param ConstraintViolationListInterface $violations
     * @return string
     */
    private function violationsToString(ConstraintViolationListInterface $violations): string
    {
        return implode(
            ' ',
            array_map(
                function (ConstraintViolationInterface $violation) {
                    return $violation->getMessage();
                },
                iterator_to_array($violations)
            )
        );
    }

    /**
     * @param DataObject\AbstractObject $taxon
     * @param string $url
     * @return bool
     */
    private function execute(DataObject\AbstractObject $taxon, Taxon $taxonData): bool
    {
        $urlProperty = $this->taxonConfiguration->getUrlPropertyName();

        foreach ($taxonData->getTaxonTranslations() as $taxonTranslationData) {
            $taxon->set($urlProperty, $taxonTranslationData->getUrl(), $taxonTranslationData->getLocale());
        }

        $taxon->save(['versionNote' => 'Auto update by external change.']);

        return true;
    }
}
