<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\Translation\Handler;

use Doctrine\Common\Persistence\ObjectManager;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Pimcore\Translation\Configuration\TranslationConfiguration;
use Lifestyle\Pimcore\Translation\Exception\InvalidPropertyException;
use Lifestyle\Pimcore\Translation\Translator\AttributeTranslator;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Pimcore\Model\DataObject;
use Psr\Log\LoggerInterface;

/**
 * Class TranslationHandler
 *
 * Receives messages from rabbitMQ, searches and saves dependent objects.
 * Translation of attribute keys and values itself is done in pimcore-export bundle
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\Translation\Handler
 */
class TranslationHandler implements HandlerInterface
{
    const PROPERTY_OBJECT_ID = 'id';
    const PROPERTY_SEARCH_KEY_NAME = 'selector';

    /**
     * @var \Pimcore\Db\Connection
     */
    public $db;

    /**
     * @var TranslationConfiguration
     */
    private $translationConfiguration;

    /**
     * @var AttributeTranslator
     */
    private $attributeTranslator;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * TranslationHandler constructor.
     * @param \Pimcore\Db\Connection $db
     * @param TranslationConfiguration $translationConfiguration
     * @param AttributeTranslator $attributeTranslator
     * @param LoggerInterface $logger
     */
    public function __construct(
        \Pimcore\Db\Connection $db,
        TranslationConfiguration $translationConfiguration,
        AttributeTranslator $attributeTranslator,
        LoggerInterface $logger
    ) {
        $this->db = $db;
        $this->translationConfiguration = $translationConfiguration;
        $this->attributeTranslator = $attributeTranslator;
        $this->logger = $logger;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        // Happily delete the pimcore runtime cache - makes life easier
        \Pimcore\Cache\Runtime::clear();
        $this->attributeTranslator->clearCache();

        // Reconnect to database
        if ($this->db->ping() === false) {
            $this->db->close();
            $this->db->connect();
        }

        $translationObjectId = $this->getTranslationObjectId($dataCollector);
        $searchKey = $this->getSearchKey($dataCollector);
        $objectIds = $this->getObjectIds($searchKey, $translationObjectId);

        $this->logger->debug(sprintf(
            'Start processing %d objects for attribute translation.',
            count($objectIds)
        ));

        $skippedObjects = 0;
        $processedObjects = 0;
        foreach ($objectIds as $objectId) {
            $object = DataObject::getById($objectId);
            if (
                !$object instanceof DataObject\Concrete ||
                !in_array($object->getClassId(), $this->translationConfiguration->getTranslatableDataObjectClassIds())
            ) {
                ++$skippedObjects;
                $this->logger->debug(sprintf(
                    'Skipping object %d! Not a translatable object.',
                    $objectId
                ));
                continue;
            }

            // Object save will trigger export
            $object->save();
            ++$processedObjects;
        }

        $this->logger->debug(sprintf(
            'Finished processing %d objects for attribute translation (processed %d, skipped %d)',
            count($objectIds),
            $processedObjects,
            $skippedObjects
        ));

        return true;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return in_array(
            $dataCollector->getItemValue('event'),
            $this->translationConfiguration->getResponsibleEvents()
        );
    }

    /**
     * @param $searchKey
     * @param $translationObjectId
     * @return array
     * @throws \Doctrine\DBAL\DBALException
     */
    private function getObjectIdsByAttributeValue($searchKey, $translationObjectId): array
    {
        $objectIds = [];
        foreach ($this->translationConfiguration->getTranslatableDataObjectClassIds() as $classId) {
            $select = $this->db->select();
            $select->from(
                ['object_classificationstore_data_' . $classId],
                ['o_id']
            );
            $select->where(
                'value=?',
                $searchKey
            );
            $objectIds = array_merge(
                $objectIds,
                array_map('intval', $this->db->fetchCol($select))
            );
        }

        return $objectIds;
    }

    /**
     * @param $searchKey
     * @throws \Doctrine\DBAL\DBALException
     */
    private function getObjectClassificationStoreKeys($searchKey): array
    {
        $select = $this->db->select();
        $select->from(
            ['classificationstore_keys'],
            ['id']
        );
        $select->where(
            'name=?',
            $searchKey
        );

        return array_map('intval', $this->db->fetchCol($select));
    }

    /**
     * @param $searchKey
     * @return mixed
     * @throws \Doctrine\DBAL\DBALException
     */
    private function getObjectIdsByAttriubteKey($searchKey): array
    {
        $keyIds = $this->getObjectClassificationStoreKeys($searchKey);
        if (0 === count($keyIds)) {
            return [];
        }

        $objectIds = [];
        foreach ($this->translationConfiguration->getTranslatableDataObjectClassIds() as $classId) {
            $select = $this->db->select();
            $select->from(
                ['object_classificationstore_data_' . $classId],
                ['o_id']
            );
            $select->where($select->expr()->in('keyId', ':keyIds'));
            $select->setParameter('keyIds', $keyIds, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY);
            $objectIds = array_merge(
                $objectIds,
                array_map('intval', $select->execute()->fetchAll(\Doctrine\DBAL\FetchMode::COLUMN))
            );
        }

        return $objectIds;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return int
     */
    private function getTranslationObjectId(DataCollectorInterface $dataCollector): int
    {
        $translationObjectId = (int)$dataCollector->getItemValue(self::PROPERTY_OBJECT_ID);
        if (0 === $translationObjectId) {
            throw new InvalidPropertyException(sprintf(
                'Cannot process translation message! Property "%s" not found in message.',
                self::PROPERTY_OBJECT_ID
            ));
        }
        return $translationObjectId;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return string
     */
    private function getSearchKey(DataCollectorInterface $dataCollector): string
    {
        $searchKey = (string)$dataCollector->getItemValue(self::PROPERTY_SEARCH_KEY_NAME);
        if ('' === $searchKey) {
            throw new InvalidPropertyException(sprintf(
                'Cannot process translation message! Property "%s" not found in message.',
                self::PROPERTY_SEARCH_KEY_NAME
            ));
        }
        return $searchKey;
    }

    /**
     * @param string $searchKey
     * @param int $translationObjectId
     * @return array
     * @throws \Doctrine\DBAL\DBALException
     */
    private function getObjectIds(string $searchKey, int $translationObjectId): array
    {
        return array_filter(array_merge(
            $this->getObjectIdsByAttriubteKey($searchKey),
            $this->getObjectIdsByAttributeValue($searchKey, $translationObjectId)
        ), function ($objectId) use ($translationObjectId) {
            // Prevent from infinite loops - remove current id
            return $objectId !== $translationObjectId;
        });
    }
}
