<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Pimcore\Translation\Translator;

use Lifestyle\Pimcore\Translation\Configuration\TranslationConfiguration;
use Pimcore\Model\DataObject;
use Symfony\Component\Translation\Translator;
use Symfony\Component\Translation\TranslatorInterface;
use Symfony\Component\Translation\Loader\ArrayLoader;

/**
 * Class AttributeTranslator
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Pimcore\Translation\Translator
 */
class AttributeTranslator implements TranslatorInterface
{
    /**
     * @var TranslationConfiguration
     */
    private $translationConfiguration;

    /**
     * @var string
     */
    private $defaultLocale;

    /**
     * @var DataObject\Translator[]
     */
    private $translators;

    /**
     * AttributeTranslator constructor.
     * @param TranslationConfiguration $translationConfiguration
     */
    public function __construct(TranslationConfiguration $translationConfiguration)
    {
        $this->translationConfiguration = $translationConfiguration;
        $this->defaultLocale = $translationConfiguration->getDefaultLocale();
        $this->translators = [];
    }

    /**
     * Delete translators from memory
     */
    public function clearCache(): void
    {
        $this->translators = [];
    }

    /**
     * Translates the given message.
     *
     * @param string $id The message id (may also be an object that can be cast to string)
     * @param array $parameters An array of parameters for the message
     * @param string|null $domain The domain for the message or null to use the default
     * @param string|null $locale The locale or null to use the default
     *
     * @return string The translated string
     *
     * @throws \InvalidArgumentException If the locale contains invalid characters
     */
    public function trans($id, array $parameters = array(), $domain = null, $locale = null)
    {
        return $this->getTranslatorByLocale($domain, $locale)->trans($id, $parameters, $domain, $locale);
    }

    /**
     * Translates the given choice message by choosing a translation according to a number.
     *
     * @param string      $id         The message id (may also be an object that can be cast to string)
     * @param int         $number     The number to use to find the index of the message
     * @param array       $parameters An array of parameters for the message
     * @param string|null $domain     The domain for the message or null to use the default
     * @param string|null $locale     The locale or null to use the default
     *
     * @return string The translated string
     *
     * @throws InvalidArgumentException If the locale contains invalid characters
     */
    public function transChoice($id, $number, array $parameters = array(), $domain = null, $locale = null)
    {
        return $this->getTranslatorByLocale($domain, $locale)->transChoice($id, $number, $parameters, $domain, $locale);
    }

    /**
     * Sets the current locale.
     *
     * @param string $locale The locale
     *
     * @throws InvalidArgumentException If the locale contains invalid characters
     */
    public function setLocale($locale)
    {
        $this->defaultLocale = $locale;
    }

    /**
     * Returns the current locale.
     *
     * @return string The locale
     */
    public function getLocale()
    {
        return $this->defaultLocale;
    }

    /**
     * @param string $locale
     * @return Translator
     */
    private function getTranslatorByLocale($domain = null, $locale = null): Translator
    {
        $locale = $locale ?: $this->defaultLocale;
        if (!isset($this->translators[$locale])) {
            $translator = new Translator($locale);
            $translator->addLoader('array', new ArrayLoader());
            $translations = [];
            $translationClass = $this->translationConfiguration->getTranslationDataObjectClassName();
            foreach ($translationClass::getByDomain($domain ?: 'messages') as $translation) {
                $translations[$translation->getSelector()] = $translation->getTranslation($locale);
            }
            $translator->addResource('array', $translations, $locale, $domain);
            $this->translators[$locale] = $translator;
        }
        return $this->translators[$locale];
    }
}
