<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg,Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ProtocolServiceSdk\Service;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use JMS\Serializer\SerializerInterface;
use LifeStyle\ProtocolServiceSdk\Exception\ProtocolServiceSdkException;
use LifeStyle\ProtocolServiceSdk\Model\Protocol;
use LifeStyle\ProtocolServiceSdk\Model\ProtocolCriteria;
use LifeStyle\ProtocolServiceSdk\Model\ProtocolLine;
use LifeStyle\ProtocolServiceSdk\Model\ProtocolList;
use Psr\Http\Client\ClientExceptionInterface;

/**
 * Class ProtocolService
 * @package LifeStyle\ProtocolServiceSdk\Service
 */
class ProtocolService
{
    /**
     * @var ClientInterface
     */
    private $httpClient;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * ProtocolService constructor.
     * @param SerializerInterface $serializer
     * @param string|null $baseUri
     * @throws Exception
     */
    public function __construct(SerializerInterface $serializer, ?string $baseUri = null)
    {
        $this->httpClient = $this->createClient($baseUri);
        $this->serializer = $serializer;

    }

    /**
     * @param string|null $baseUri
     * @return Client
     * @throws Exception
     */
    protected function createClient(?string $baseUri): Client
    {
        if (null === $baseUri) {
            throw new ProtocolServiceSdkException('Base URI missing. You can configure this in the yaml file.');
        }

        return new Client([
            'base_uri' => $baseUri,
        ]);
    }

    /**
     * @param int $protocolId
     * @return Protocol
     * @throws ProtocolServiceSdkException
     */
    public function read(int $protocolId): Protocol
    {
        try {
            $response = $this->httpClient->request('GET', "/protocol/{$protocolId}", []);

            return $this->serializer->deserialize((string)$response->getBody(), Protocol::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::withGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::withOtherException($exception);
        }
    }

    /**
     * @param ProtocolCriteria $criteria
     * @return ProtocolList
     * @throws ProtocolServiceSdkException
     */
    public function list(ProtocolCriteria $criteria): ProtocolList
    {
        try {
            $response = $this->httpClient->request('GET', '/protocol/list', [
                'body' => $this->serializer->serialize($criteria, 'json'),
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), ProtocolList::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::withGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::withOtherException($exception);
        }
    }

    /**
     * @param Protocol $protocol
     * @return Protocol
     * @throws ProtocolServiceSdkException
     */
    public function create(Protocol $protocol): Protocol
    {
        try {
            $response = $this->httpClient->request('POST', '/protocol/', [
                'body' => $this->serializer->serialize($protocol, 'json'),
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), Protocol::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::withGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::withOtherException($exception);
        }
    }

    /**
     * @param Protocol $protocol
     * @param ProtocolLine $protocolLine
     * @return ProtocolLine
     * @throws ProtocolServiceSdkException
     */
    public function createLine(Protocol $protocol, ProtocolLine $protocolLine): ProtocolLine
    {
        try {
            $response = $this->httpClient->request('POST', "/protocol/{$protocol->getId()}/line", [
                'body' => $this->serializer->serialize($protocolLine, 'json'),
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), ProtocolLine::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::withGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::withOtherException($exception);
        }
    }
}
