<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace LifeStyle\Tools\SamlBundle\Saml;

use LifeStyle\Tools\SamlBundle\Exception\InvalidSamlConfigurationException;

/**
 * Class Configuration
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package LifeStyle\Tools\SamlBundle\Saml
 */
class Configuration
{
    /**
     * @var string
     */
    private $idpPublicCertificateFile;

    /**
     * @var string
     */
    private $idpSingleSignOnUrl;

    /**
     * @var string
     */
    private $idpLogoutUrl;

    /**
     * @var string
     */
    private $idpRequestPrefix;

    /**
     * @var string
     */
    private $spReturnUrl;

    /**
     * @var string
     */
    private $spIssuer;

    /**
     * @var string
     */
    private $requestedNameIdFormat;

    /**
     * @var string
     */
    private $cookiePath;

    /**
     * @var string
     */
    private $idpPublicCertificate;

    /**
     * Configuration constructor.
     * @param string $idpPublicCertificateFile
     * @param string $idpSingleSignOnUrl
     * @param string $idpLogoutUrl
     * @param string $idpRequestPrefix
     * @param string $spReturnUrl
     * @param string $spIssuer
     * @param string $requestedNameIdFormat
     * @param string $cookiePath
     */
    public function __construct(
        string $idpPublicCertificateFile,
        string $idpSingleSignOnUrl,
        string $idpLogoutUrl,
        string $idpRequestPrefix,
        string $spReturnUrl,
        string $spIssuer,
        string $requestedNameIdFormat,
        string $cookiePath
    ) {
        $this->idpPublicCertificateFile = $idpPublicCertificateFile;
        $this->idpSingleSignOnUrl = $idpSingleSignOnUrl;
        $this->idpLogoutUrl = $idpLogoutUrl;
        $this->idpRequestPrefix = $idpRequestPrefix;
        $this->spReturnUrl = $spReturnUrl;
        $this->spIssuer = $spIssuer;
        $this->requestedNameIdFormat = $requestedNameIdFormat;
        $this->cookiePath = $cookiePath;
    }

    /**
     * @return string
     */
    public function getIdpPublicCertificateFile(): string
    {
        return $this->idpPublicCertificateFile;
    }

    /**
     * @return string
     */
    public function getIdpSingleSignOnUrl(): string
    {
        return $this->idpSingleSignOnUrl;
    }

    /**
     * @return string
     */
    public function getIdpLogoutUrl(): string
    {
        return $this->idpLogoutUrl;
    }

    /**
     * @return string
     */
    public function getIdpRequestPrefix(): string
    {
        return $this->idpRequestPrefix;
    }

    /**
     * @return string
     */
    public function getSpReturnUrl(): string
    {
        return $this->spReturnUrl;
    }

    /**
     * @return string
     */
    public function getSpIssuer(): string
    {
        return $this->spIssuer;
    }

    /**
     * @return string
     */
    public function getRequestedNameIdFormat(): string
    {
        return $this->requestedNameIdFormat;
    }

    /**
     * @return string
     */
    public function getCookiePath(): string
    {
        return $this->cookiePath;
    }

    /**
     * @return string
     * @throws InvalidSamlConfigurationException
     */
    public function getIdpPublicCertificate(): string
    {
        if (null === $this->idpPublicCertificate) {
            if (!file_exists($this->idpPublicCertificateFile)) {
                throw new InvalidSamlConfigurationException(
                    'Missing idp SAML certification file at ' . $this->idpPublicCertificateFile
                );
            }
            $this->idpPublicCertificate = file_get_contents($this->idpPublicCertificateFile);
        }
        return $this->idpPublicCertificate;
    }
}
