<?php
/**
 * Saml worker
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Tools\SamlBundle\Saml;

use LifeStyle\Tools\SamlBundle\Library\Utils\Security;
use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use LifeStyle\Tools\SamlBundle\Exception\InvalidSamlConfigurationException;
use LifeStyle\Tools\SamlBundle\Event\FilterCookieFileEvent;
use LifeStyle\Tools\SamlBundle\Event\CookieFile as CookieFileEvent;

/**
 * Saml worker
 */
class Worker
{

    /**
     * Symfony logger
     *
     * @var Logger
     */
    protected $logger;

    /**
     * Saml request builder
     *
     * @var Request
     */
    protected $request;

    /**
     * Saml-Response-Handler
     *
     * @var Handler
     */
    protected $handler;

    /**
     * Saml config
     *
     * @var Configuration
     */
    protected $samlConfig;

    /**
     * Cookie filename
     *
     * @var string
     */
    protected $cookieFile;

    /**
     * Event Dispatcher
     *
     * @var EventDispatcherInterface
     */
    protected $eventDispatcher;

    /**
     * Event filter
     *
     * @var FilterCookieFileEvent
     */
    protected $eventFilter;

    /**
     * Constructor
     *
     * @param Logger $logger
     * @param EventDispatcherInterface $eventDispatcher
     * @param Configuration $samlConfig SAML configuration
     */
    public function __construct(Logger $logger, EventDispatcherInterface $eventDispatcher, Configuration $samlConfig)
    {
        $this->logger = $logger;
        $this->eventDispatcher = $eventDispatcher;
        $this->samlConfig = $samlConfig;
    }

    /**
     * Destructor
     */
    public function __destruct()
    {
        $this->eventDispatcher->dispatch(CookieFileEvent::SHUTDOWN, $this->filterCookieFileEvent());
    }

    /**
     * Login user
     *
     * @param string $username
     * @param string $password
     * @return bool True on success
     */
    public function login($username, $password)
    {
        // Try to get login url from check request
        if (!($loginUrl = $this->handler()->getLoginUrl())) {
            $this->check($username);
            if (!($loginUrl = $this->handler()->getLoginUrl())) {
                return false;
            }
        }

        $post = http_build_query(array(
            'username' => $username,
            'password' => $password,
            'firstname' => Security::stringRewinder($username),
            'middlename' => '',
            'lastname' => Security::stringNiner($username),
            'comments' => Security::getSessionIdFromCookie($this->getCookieFile())
        ));

        // Send get-request
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
        curl_setopt($ch, CURLOPT_URL, $loginUrl);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        curl_setopt($ch, CURLOPT_COOKIEJAR, $this->getCookieFile());
        curl_setopt($ch, CURLOPT_COOKIEFILE, $this->getCookieFile());
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        curl_close($ch);

        $loggedIn = $this->handler()->login($response);
        if ($loggedIn) {
            $this->saveCookie();
        } else {
            $this->removeCookie();
        }

        return $loggedIn;
    }

    /**
     * Logout current user
     * @param $username
     * @return bool|void
     * @throws InvalidSamlConfigurationException
     */
    public function logout($username)
    {
        // Try to get last saml response
        if (!($samlResponse = $this->handler()->getSamlResponse())) {
            $this->check($username);
            if (!($samlResponse = $this->handler()->getSamlResponse())) {
                return false;
            }
        }

        $logoutUrl = $this->request()->getLogoutUrl($samlResponse);

        // Send request - follow redirects
        $redirects = 0;
        do {
            if (++$redirects > 100) {
                return false;
            }
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
            curl_setopt($ch, CURLOPT_URL, $logoutUrl);
            curl_setopt($ch, CURLOPT_HEADER, true);
            curl_setopt($ch, CURLOPT_COOKIEJAR, $this->getCookieFile());
            curl_setopt($ch, CURLOPT_COOKIEFILE, $this->getCookieFile());
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $response = curl_exec($ch);
            curl_close($ch);
        } while (preg_match('#Location: (.*)#', $response, $matches) && ($logoutUrl = trim($matches[1])));

        $this->removeCookie();
        return $this->handler()->logout($response);
    }

    /**
     * Check current user logged in
     *
     * @param string $username
     * @return bool
     */
    public function check($username)
    {
        $url = $this->request()->getRedirectUrl();

        // Send request
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_COOKIEJAR, $this->getCookieFile());
        curl_setopt($ch, CURLOPT_COOKIEFILE, $this->getCookieFile());
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        $response = curl_exec($ch);
        curl_close($ch);

        $loggedIn = $this->handler()->check($response, $username);

        return $loggedIn;
    }

    /**
     * Get attribute from saml response
     *
     * @param string $name
     * @return mixed
     */
    public function getAttribute($name)
    {
        return $this->handler()->getAttribute($name);
    }

    /**
     * Session lifetime as unix timestamp
     *
     * @return integer|null
     */
    public function getSessionLifetime()
    {
        return $this->handler()->getSamlResponse()->getSessionNotOnOrAfter();
    }

    /**
     * Get cookie filename
     *
     * @return string
     * @throws InvalidSamlConfigurationException
     */
    protected function getCookieFile()
    {
        if (null === $this->cookieFile) {
            $this->eventDispatcher->dispatch(CookieFileEvent::LOAD, $this->filterCookieFileEvent());
            $this->cookieFile = $this->filterCookieFileEvent()->getCookieFile();
            if (empty($this->cookieFile)) {
                throw new InvalidSamlConfigurationException('Missing SAML cookie filename!');
            }
        }
        return $this->cookieFile = $this->filterCookieFileEvent()->getCookieFile();
    }

    /**
     * Save cookie file
     */
    protected function saveCookie()
    {
        $this->eventDispatcher->dispatch(CookieFileEvent::SAVE, $this->filterCookieFileEvent());
        $this->cookieFile = null;
    }

    /**
     * Remove cookie file
     */
    protected function removeCookie()
    {
        $this->eventDispatcher->dispatch(CookieFileEvent::REMOVE, $this->filterCookieFileEvent());
        $this->cookieFile = null;
    }

    /**
     * Cookie event filter
     *
     * @return FilterCookieFileEvent
     */
    protected function filterCookieFileEvent()
    {
        return null !== $this->eventFilter ? $this->eventFilter : ($this->eventFilter = new FilterCookieFileEvent($this->samlConfig));
    }

    /**
     * Saml request builder
     *
     * @return Request
     */
    protected function request()
    {
        return null !== $this->request ? $this->request : ($this->request = new Request($this->logger, $this->samlConfig));
    }

    /**
     * Response handler
     *
     * @return Handler
     */
    protected function handler()
    {
        return null !== $this->handler ? $this->handler : ($this->handler = new Handler($this->logger, $this->samlConfig));
    }
}
