<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace SimpleSAML\Module\lifestyle\Auth\Source;

use SimpleSAML\Configuration;
use SimpleSAML\Error\Error;
use SimpleSAML\Error\Exception;
use SimpleSAML\Logger;
use SimpleSAML\Module\core\Auth\UserPassBase;
use SimpleSAML\Module\lifestyle\Utils\Request;
use SimpleSAML\Module\lifestyle\Utils\Security;
use SimpleSAML\Utils\Crypto;

/**
 * Class AdminPassword
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package SimpleSAML\Module\lifestyle\Auth\Source
 */
class AdminPassword extends UserPassBase
{
    /**
     * Constructor for this authentication source.
     *
     * @param array $info Information about this authentication source.
     * @param array $config Configuration.
     */
    public function __construct($info, $config)
    {
        assert('is_array($info)');
        assert('is_array($config)');

        // Call the parent constructor first, as required by the interface
        parent::__construct($info, $config);
    }

    /**
     * Attempt to log in using the given username and password.
     *
     * On a successful login, this function should return the users attributes. On failure,
     * it should throw an exception. If the error was caused by the user entering the wrong
     * username or password, a SimpleSAML\Error\Error('WRONGUSERPASS') should be thrown.
     *
     * Note that both the username and the password are UTF-8 encoded.
     *
     * @param string $username
     * @param string $password
     * @return array
     * @throws \Exception
     * @throws Error
     * @throws Exception
     */
    protected function login($username, $password)
    {
        assert('is_string($username)');
        assert('is_string($password)');

        // Captcha replacement - check first
        if (true !== Security::validate($username, $_POST)) {
            Logger::stats('Login failure for admin user ' . $username . ' from ip ' . Request::getClientIp() . '. Captcha validation failed.');
            throw new Error('WRONGUSERPASS');
        }

        $config = Configuration::getInstance();
        $adminPassword = $config->getString('auth.adminpassword', '123');
        if ($adminPassword === '123') {
            // We require that the user changes the password
            throw new Error('NOTSET');
        }

        $adminUsername = $config->getString('auth.adminusername', 'admin');
        if ($username !== $adminUsername) {
            Logger::stats('Login failure for admin user ' . $username . ' from ip ' . Request::getClientIp() . '. Invalid username.');
            throw new Error('WRONGUSERPASS');
        }

        if (!Crypto::pwValid($adminPassword, $password)) {
            Logger::stats('Login failure for admin user ' . $username . ' from ip ' . Request::getClientIp() . '. Invalid password.');
            throw new Error('WRONGUSERPASS');
        }

        return array('user' => array('admin'));
    }
}
