<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace SimpleSAML\Module\lifestyle\Utils;

use SimpleSAML\Auth\State;
use SimpleSAML\Configuration;
use SimpleSAML\Metadata\MetaDataStorageHandler;
use SimpleSAML\Utils\HTTP;

/**
 * Class UrlBuilder
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package SimpleSAML\Module\lifestyle\Utils
 */
class UrlBuilder
{
    /**
     * @var Configuration
     */
    private $configuration;

    /**
     * @var string
     */
    private $authState;

    /**
     * @var string
     */
    private $language;

    /**
     * \SimpleSAML\Module\lifestyle\Utils\UrlBuilder constructor.
     * @param Configuration $configuration
     * @param string $authState
     * @param string $language
     */
    public function __construct(Configuration $configuration, $authState, $language)
    {
        $this->configuration = $configuration;
        $this->authState = $authState;
        $this->language = $language;
    }

    /**
     * @return string
     */
    public function getPasswordResetUrl()
    {
        return $this->getBaseUrl('lifestyle.password.reset.url');
    }

    /**
     * @return string
     */
    public function getPasswordForgottenUrl()
    {
        $url = $this->getBaseUrl('lifestyle.password.forgotten.url');

        $authState = State::parseStateID($this->authState);
        if (empty($authState['url'])) {
            return $url;
        }

        $query = parse_url($authState['url'], PHP_URL_QUERY);
        if (empty($authState['url'])) {
            return $url;
        }

        parse_str($query, $queryParams);
        if (empty($queryParams['spentityid'])) {
            return $url;
        }
        $queryParams = array_intersect_key($queryParams, array_flip(['spentityid', 'cookieTime', 'RelayState']));

        /** @var SimpleSAML\Metadata\MetaDataStorageHandler $metadata */
        $metadata = MetaDataStorageHandler::getMetadataHandler();
        $returnBaseUrl = $metadata->getGenerated('SingleSignOnService', 'saml20-idp-hosted');
        $returnUrl = HTTP::addURLParameters($returnBaseUrl, $queryParams);

        return \SimpleSAML\Utils\HTTP::addURLParameters($url, ['idpreturnurl' => $returnUrl]);
    }

    /**
     * @return string
     */
    public function getMfaForgottenUrl()
    {
        return $this->getBaseUrl('lifestyle.mfa.forgotten.url');
    }

    /**
     * @return string
     */
    public function getPrivacyUrl()
    {
        return $this->getBaseUrl('lifestyle.privacy.url');
    }

    /**
     * @return string
     */
    public function getSsoBaseUrl()
    {
        return $this->getBaseUrl('lifestyle.sso.baseurl');
    }

    /**
     * @param string $key
     * @return string
     */
    private function getBaseUrl($key)
    {
        $urls = $this->configuration->getArray($key);
        return isset($urls[$this->language]) ? $urls[$this->language] : reset($urls);
    }
}
