<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace SimpleSAML\Module\lifestyle\Webservice\Model\Response\IdP\PostLogin;

use DateTime;
use JMS\Serializer\Annotation as Serializer;
use SimpleSAML\Module\lifestyle\Webservice\Model\Response\Errors;

/**
 * Class Response
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package SimpleSAML\Module\lifestyle\Webservice\Model\Response\IdP\PostLogin
 * @Serializer\XmlRoot("response")
 */
class Response
{
    const STATUS_VERIFIED = 'verified';

    /**
     * @Serializer\SerializedName("status")
     * @Serializer\Type("integer")
     *
     * @var integer
     */
    private $status;

    /**
     * @Serializer\SerializedName("message")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $message;

    /**
     * @Serializer\SerializedName("trackId")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $trackId;

    /**
     * @Serializer\SerializedName("durationSeconds")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $durationSeconds;

    /**
     * @Serializer\SerializedName("loginStatus")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $loginStatus;

    /**
     * @Serializer\SerializedName("userGuid")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userGuid;

    /**
     * @Serializer\SerializedName("userIdentifier")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userIdentifier;

    /**
     * @Serializer\SerializedName("userAuthId")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userAuthId;

    /**
     * @Serializer\SerializedName("userLdapSearchAttributes")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userLdapSearchAttributes;

    /**
     * @Serializer\SerializedName("userLdapSearchValue")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userLdapSearchValue;

    /**
     * @Serializer\SerializedName("userEmail")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userEmail;

    /**
     * @Serializer\SerializedName("userFirstname")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userFirstname;

    /**
     * @Serializer\SerializedName("userLastname")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $userLastname;

    /**
     * @Serializer\SerializedName("mfaEnabled")
     * @Serializer\Type("boolean")
     *
     * @var boolean
     */
    private $mfaEnabled;

    /**
     * @Serializer\SerializedName("mfaQrCode")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $mfaQrCode;

    /**
     * @Serializer\SerializedName("forcePasswordReset")
     * @Serializer\Type("boolean")
     *
     * @var boolean
     */
    private $forcePasswordReset = false;

    /**
     * @Serializer\SerializedName("resetPasswordToken")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $resetPasswordToken;

    /**
     * @Serializer\SerializedName("resetPasswordTokenExpire")
     * @Serializer\Type("DateTime")
     *
     * @var DateTime
     */
    private $resetPasswordTokenExpire;

    /**
     * @Serializer\SerializedName("lastLogin")
     * @Serializer\Type("DateTime")
     *
     * @var string
     */
    private $lastLogin;

    /**
     * @Serializer\SerializedName("loginFails")
     * @Serializer\Type("integer")
     *
     * @var integer
     */
    private $loginFails;

    /**
     * @Serializer\SerializedName("errors")
     * @Serializer\Type("SimpleSAML\Module\lifestyle\Webservice\Model\Response\Errors")
     *
     * @var Errors
     */
    private $errors;

    /**
     * @return int
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * @return string
     */
    public function getMessage()
    {
        return $this->message;
    }

    /**
     * @return string
     */
    public function getTrackId()
    {
        return $this->trackId;
    }

    /**
     * @return string
     */
    public function getDurationSeconds()
    {
        return $this->durationSeconds;
    }

    /**
     * @return string
     */
    public function getLoginStatus()
    {
        return $this->loginStatus;
    }

    /**
     * @return bool
     */
    public function isUserVerified()
    {
        return $this->loginStatus === static::STATUS_VERIFIED;
    }

    /**
     * @return string
     */
    public function getUserIdentifier()
    {
        return $this->userIdentifier;
    }

    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->userIdentifier;
    }

    /**
     * @return string
     */
    public function getUserGuid()
    {
        return $this->userGuid;
    }

    /**
     * @return string
     */
    public function getUserAuthId()
    {
        return $this->userAuthId;
    }

    /**
     * @return string
     */
    public function getUserLdapSearchAttributes()
    {
        return $this->userLdapSearchAttributes;
    }

    /**
     * @return string
     */
    public function getUserLdapSearchValue()
    {
        return $this->userLdapSearchValue;
    }

    /**
     * @return string
     */
    public function getUserEmail()
    {
        return $this->userEmail;
    }

    /**
     * @return string
     */
    public function getUserFirstname()
    {
        return $this->userFirstname;
    }

    /**
     * @return string
     */
    public function getUserLastname()
    {
        return $this->userLastname;
    }

    /**
     * @return boolean
     */
    public function isMfaEnabled()
    {
        return $this->mfaEnabled;
    }

    /**
     * @return string
     */
    public function getMfaQrCode()
    {
        return $this->mfaQrCode;
    }

    /**
     * @return bool
     */
    public function isForcePasswordReset()
    {
        return $this->forcePasswordReset;
    }

    /**
     * @return string
     */
    public function getResetPasswordToken()
    {
        return $this->resetPasswordToken;
    }

    /**
     * @return DateTime
     */
    public function getResetPasswordTokenExpire()
    {
        return $this->resetPasswordTokenExpire;
    }

    /**
     * @return string
     */
    public function getLastLogin()
    {
        return $this->lastLogin;
    }

    /**
     * @return int
     */
    public function getLoginFails()
    {
        return $this->loginFails;
    }

    /**
     * @return bool
     */
    public function isPasswordValid()
    {
        return null === $this->errors && $this->isUserVerified();
    }

    /**
     * @return Errors|null
     */
    public function getErrors()
    {
        return $this->errors;
    }

    /**
     * @return array
     */
    public function getUserAttributes()
    {
        $attributes = array();
        foreach ($this->getAttributeMap() as $source => $target) {
            if (strlen($this->$source)) {
                $attributes[$target][] = $this->$source;
            }
        }
        return $attributes;
    }

    /**
     * @return array
     */
    private function getAttributeMap()
    {
        return array(
            'userGuid' => 'guid',
            'userIdentifier' => 'username',
            'userEmail' => 'email',
            'userFirstname' => 'firstname',
            'userLastname' => 'lastname',
            'mfaEnabled' => 'mfaEnabled',
            'mfaQrCode' => 'mfaQrCode',
        );
    }
}
