<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace SimpleSAML\Module\lifestyle\Auth\Source;

use SAML2\Constants;
use SimpleSAML\Auth;
use SimpleSAML\Error;
use SimpleSAML\Logger;
use SimpleSAML\Module;
use SimpleSAML\Module\lifestyle\Webservice\Handler;
use SimpleSAML\Module\saml\Auth\Source\SP;
use SimpleSAML\Session;
use SimpleSAML\Utils\HTTP;

/**
 * Class UserPassBase
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package SimpleSAML\Module\lifestyle\Auth\Source
 */
abstract class UserPassBase extends \SimpleSAML\Module\core\Auth\UserPassBase
{
    /**
     * @var bool
     */
    private static $usernameValid = false;

    /**
     * Initialize login.
     *
     * This function saves the information about the login, and redirects to a
     * login page.
     *
     * @param array &$state  Information about the current authentication.
     * @return void
     */
    public function authenticate(&$state)
    {
        assert(is_array($state));

        /*
         * Save the identifier of this authentication source, so that we can
         * retrieve it later. This allows us to call the login()-function on
         * the current object.
         */
        $state[self::AUTHID] = $this->authId;

        // ECP requests supply authentication credentials with the AuthnRequest
        // so we validate them now rather than redirecting. The SAML spec
        // doesn't define how the credentials are transferred, but Office 365
        // uses the Authorization header, so we will just use that in lieu of
        // other use cases.
        if (isset($state['saml:Binding']) && $state['saml:Binding'] === Constants::BINDING_PAOS) {
            if (!isset($_SERVER['PHP_AUTH_USER']) || !isset($_SERVER['PHP_AUTH_PW'])) {
                Logger::error("ECP AuthnRequest did not contain Basic Authentication header");
                // TODO Return a SOAP fault instead of using the current binding?
                throw new Error\Error("WRONGUSERPASS");
            }

            $username = $_SERVER['PHP_AUTH_USER'];
            $password = $_SERVER['PHP_AUTH_PW'];

            if (isset($state['forcedUsername'])) {
                $username = $state['forcedUsername'];
            }

            $attributes = $this->login($username, $password);
            assert(is_array($attributes));
            $state['Attributes'] = $attributes;

            return;
        }

        // Save the $state-array, so that we can restore it after a redirect
        $id = Auth\State::saveState($state, self::STAGEID);

        /*
         * Redirect to the login form. We include the identifier of the saved
         * state array as a parameter to the login form.
         */
        $url = Module::getModuleURL('lifestyle/loginuserpass.php');
        $params = ['AuthState' => $id];
        HTTP::redirectTrustedURL($url, $params);

        // The previous function never returns, so this code is never executed.
        assert(false);
    }

    /**
     * @param array $state
     * @param string $username
     */
    public static function validateUsername(array $state, string $username)
    {
        self::$usernameValid = false;
        $user = Handler::showUser($username);
        if (null !== $user && 0 < strlen($user->getUserAuthId())) {

            $source = null;
            try {
                $source = Auth\Source::getById($user->getUserAuthId(), SP::class);
            } catch (Error\Exception $exception) {
                unset($exception);
            }

            // Authenticate against another idp
            if (null !== $source) {
                Session::getSessionFromRequest()->setData('string', 'IdPUserAuthId', $user->getUserAuthId());
                $state['IdpUserIdentifier'] = $user->getUserIdentifier();
                $source->authenticate($state);
            }
        }
    }

    /**
     * @param array $state
     */
    public function logout(&$state)
    {
        parent::logout($state);

        $userAuthId = Session::getSessionFromRequest()->getData('string', 'IdPUserAuthId');
        if (null !== $userAuthId) {
            $source = null;
            try {
                $source = Auth\Source::getById($userAuthId, SP::class);
            } catch (Error\Exception $exception) {
                unset($exception);
            }

            // Logout from other idp
            if (null !== $source) {
                $source->logout($state);
            }
        }
    }
}
