<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace SimpleSAML\Module\lifestyle\Utils;

use Psr\Container\ContainerInterface;
use Symfony\Component\Asset\Package;
use Symfony\Component\Asset\Packages;
use Symfony\Component\Asset\VersionStrategy\JsonManifestVersionStrategy;
use Symfony\Component\DependencyInjection\Container;
use Symfony\Component\Yaml\Yaml;
use Symfony\WebpackEncoreBundle\Asset\EntrypointLookup;
use Symfony\WebpackEncoreBundle\Asset\EntrypointLookupCollection;
use Symfony\WebpackEncoreBundle\Asset\TagRenderer;

/**
 * Class WebpackEncoreUtil
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package SimpleSAML\Module\lifestyle\Utils
 */
class WebpackEncoreUtil
{
    /**
     * @var TagRenderer
     */
    private static $tagRenderer;

    /**
     * @var Packages
     */
    private static $packages;

    /**
     * @var array
     */
    private static $configuration;

    /**
     * @param string $entryName
     * @param string|null $packageName
     * @param string $entrypointName
     * @return string
     */
    public function renderWebpackScriptTags(
        string $entryName,
        string $packageName = null,
        string $entrypointName = '_default'
    ): string {
        return $this->getTagRenderer()->renderWebpackScriptTags($entryName, $packageName, $entrypointName);
    }

    /**
     * @param string $entryName
     * @param string|null $packageName
     * @param string $entrypointName
     * @return string
     */
    public function renderWebpackLinkTags(
        string $entryName,
        string $packageName = null,
        string $entrypointName = '_default'
    ): string {
        return $this->getTagRenderer()->renderWebpackLinkTags($entryName, $packageName, $entrypointName);
    }

    /**
     * @param string $path
     * @param string|null $packageName
     * @return string
     */
    public function getAssetUrl(string $path, string $packageName = null): string
    {
        return $this->getPackages()->getUrl($path, $packageName);
    }

    /**
     * @return TagRenderer
     */
    private function getTagRenderer(): TagRenderer
    {
        if (null === self::$tagRenderer) {
            self::$tagRenderer = new TagRenderer($this->getEntrypointLookupCollection(), $this->getPackages());
        }

        return self::$tagRenderer;
    }

    /**
     * @return EntrypointLookupCollection
     */
    private function getEntrypointLookupCollection(): EntrypointLookupCollection
    {
        return new EntrypointLookupCollection($this->buildEntryPoints(), '_default');
    }

    /**
     * @return Packages
     */
    private function getPackages(): Packages
    {
        if (null === self::$packages) {
            $config = $this->getConfiguration();
            $defaultPackage = null;
            $packages = [];
            foreach ($config['webpack_encore']['builds'] as $name => $path) {
                $packages[$name] = new Package(new JsonManifestVersionStrategy($path . '/manifest.json'));
                if (null === $defaultPackage) {
                    $defaultPackage = $packages[$name];
                }
            }
            self::$packages = new Packages($defaultPackage, $packages);
        }

        return self::$packages;
    }

    /**
     * @return ContainerInterface
     */
    private function buildEntryPoints(): ContainerInterface
    {
        $config = $this->getConfiguration();
        $container = new Container();
        foreach ($config['webpack_encore']['builds'] as $name => $path) {
            $container->set($name, $this->buildEntrypointLookup($name, $path));
        }
        return $container;
    }

    /**
     * @param string $name
     * @param string $path
     * @return EntrypointLookup
     */
    private function buildEntrypointLookup(string $name, string $path): EntrypointLookup
    {
        return new EntrypointLookup(
            $path . '/entrypoints.json',
            null,
            $name,
            false
        );
    }

    /**
     * @return array
     */
    private function getConfiguration(): array
    {
        if (null === self::$configuration) {
            $projectDir = dirname(__DIR__, 4);
            self::$configuration = Yaml::parseFile(dirname(__DIR__, 4) . '/config/packages/webpack_encore.yaml');
            $builds = self::$configuration['webpack_encore']['builds'] ?? [];
            if (isset(self::$configuration['webpack_encore']['output_path'])) {
                $builds['_default'] = self::$configuration['webpack_encore']['output_path'];
            }
            foreach ($builds as $name => $path) {
                self::$configuration['webpack_encore']['builds'][$name] = str_replace('%kernel.project_dir%', $projectDir, $path);
            }
        }

        return self::$configuration;
    }
}
