<?php

namespace AppBundle\EventListener;

use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\FilterResponseEvent;

/**
 * Class UserIdentifierLogListener
 * @package AppBundle\EventListener
 */
class UserIdentifierLogListener
{
    /**
     * @var EntityManagerInterface
     */
    private $objectManager;

    /**
     * @var bool
     */
    private $isDebug;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var bool
     */
    private $enableLogger;

    /**
     * UserIdentifierLogListener constructor.
     * @param EntityManagerInterface $objectManager
     * @param bool $isDebug
     * @param LoggerInterface $logger
     * @param $enableLogger
     */
    public function __construct(
        EntityManagerInterface $objectManager,
        bool $isDebug,
        LoggerInterface $logger,
        $enableLogger
    ) {
        $this->objectManager = $objectManager;
        $this->isDebug = $isDebug;
        $this->logger = $logger;
        $this->enableLogger = $enableLogger == 1 || $enableLogger == true || $enableLogger === 'true';
    }

    /**
     * @param FilterResponseEvent $event
     * @throws \Exception
     */
    public function onKernelResponse(FilterResponseEvent $event)
    {
        if (!$this->enableLogger) {
            return;
        }

        try {
            $request = $event->getRequest();
            $requestHeaders = $request->headers;
            $response = $event->getResponse();
            if ('UserIdentifier' === $requestHeaders->get('API-CONTROLLER')) {
                $action = $requestHeaders->get('API-ACTION');
                if ('Validate' === $action) {
                    $this->logValidateAction($request, $response);
                } elseif ('Get' === $action) {
                    $this->logGetAction($request, $response);
                }
            }
            if (mt_rand(0, 100) < 10) {
                $this->logPrune();
            }
        } catch (\Exception $exception) {
            if ($this->isDebug) {
                throw $exception;
            } else {
                $this->logger->warning(
                    'Unable to log user-identifier-service: ' . $exception->getMessage(),
                    ['exception' => $exception]
                );
            }
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     */
    private function logValidateAction(Request $request, Response $response)
    {
        $sessionLifetime = $this->getText($response->getContent(), 'SessionLifetime');
        $params = [
            'request_type' => 'validate',
            'user_identifier' => substr($this->getText($request->getContent(), 'UserIdentifier'), 0, 64),
            'username' => $this->getText($request->getContent(), 'Username'),
            'session_lifetime_str' => $sessionLifetime,
            'session_lifetime' => (new \DateTime())->setTimestamp(strtotime($sessionLifetime))->format('Y-m-d H:i:s'),
            'is_successful' => $response->isSuccessful() ? '1' : '0',
            'created_at' => (new \DateTime())->format('Y-m-d H:i:s'),
        ];
        $this->objectManager->getConnection()->insert('log_user_identifier', $params);
    }

    /**
     * @param Request $request
     * @param Response $response
     */
    private function logGetAction(Request $request, Response $response)
    {
        $sessionLifetime = $this->getText($response->getContent(), 'SessionLifetime');
        $params = [
            'request_type' => 'get',
            'user_identifier' => substr($this->getText($response->getContent(), 'UserIdentifier'), 0, 64),
            'username' => $this->getText($request->getContent(), 'Username'),
            'session_lifetime_str' => $sessionLifetime,
            'session_lifetime' => (new \DateTime())->setTimestamp(strtotime($sessionLifetime))->format('Y-m-d H:i:s'),
            'is_successful' => $response->isSuccessful() ? '1' : '0',
            'created_at' => (new \DateTime())->format('Y-m-d H:i:s'),
        ];
        $this->objectManager->getConnection()->insert('log_user_identifier', $params);
    }

    /**
     * @param string $xml
     * @param string $tagName
     * @return string|null
     */
    private function getText(string $xml, string $tagName): ?string
    {
        $xml = str_replace(['<![CDATA[', ']]>'], ['', ''], $xml);
        if (false === preg_match('#<' . $tagName .'>([^<]+?)</' . $tagName .'>#s', $xml, $matches)) {
            return null;
        }

        return $matches[1];
    }

    private function logPrune()
    {
        $this->objectManager->getConnection()
            ->prepare('DELETE FROM log_user_identifier WHERE created_at < :created_at')
            ->execute(['created_at' => (new \DateTime('-2 days'))->format('Y-m-d H:i:s')]);
    }
}
