<?php
/**
 * Saml controller
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Tools\SamlBundle\Saml;

use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Saml controller
 */
class Controller implements SamlControllerInterface
{
    /**
     * Saml config
     *
     * @var Configuration
     */
    private $samlConfig;

    /**
     * Symfony logger
     *
     * @var Logger
     */
    private $logger;

    /**
     * Event Dispatcher
     *
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * Saml-Worker
     *
     * @var Worker
     */
    private $worker;

    /**
     * Controller constructor.
     * @param Configuration $samlConfig
     * @param Logger $logger
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(Configuration $samlConfig, Logger $logger, EventDispatcherInterface $eventDispatcher)
    {
        $this->samlConfig = $samlConfig;
        $this->logger = $logger;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * Login user
     *
     * @param string $username
     * @param string $password
     * @param bool $forceLogin
     * @return bool True on success
     */
    public function login($username, $password, $forceLogin = false)
    {
        $this->logger->addDebug(sprintf('Start to login user "%s" ...', $username));

        // Check if already logged in
        if (!$forceLogin && $this->worker()->check($username)) {
            $this->logger->addDebug(sprintf('No login needed. User "%s" already logged in.', $username));
            return true;
        }

        // Log in
        return $this->worker()->login($username, $password, $forceLogin);
    }

    /**
     * Logout current user
     * @param string $username
     * @return bool|void
     */
    public function logout($username)
    {
        $this->logger->addDebug('Start to logout user...');

        // Not logged in
        if (!$this->worker()->check($username)) {
            $this->logger->addDebug('No logout needed. User already logged out.');
            return true;
        }

        // Log out
        return $this->worker()->logout($username);
    }

    /**
     * Check current user logged in
     *
     * @param string $username
     * @return bool
     */
    public function check($username)
    {
        $this->logger->addDebug(sprintf('Start to check if user "%s" is logged in ...', $username));

        // Check if user is logged in
        return $this->worker()->check($username);
    }

    /**
     * Get attribute from saml response
     *
     * @param string $name
     * @return mixed
     */
    public function getAttribute($name)
    {
        return $this->worker()->getAttribute($name);
    }

    /**
     * Session lifetime as unix timestamp
     *
     * @return integer|null
     */
    public function getSessionLifetime()
    {
        return $this->worker()->getSessionLifetime();
    }

    /**
     * Saml Worker
     *
     * @return Worker
     */
    protected function worker()
    {
        return null !== $this->worker ? $this->worker : ($this->worker = new Worker($this->logger, $this->eventDispatcher, $this->samlConfig));
    }
}
