<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sso\LaunchpadBundle\Security\Firewall;

use Lifestyle\Sso\LaunchpadBundle\Security\Configuration\SamlConfig;
use Lifestyle\Sso\LaunchpadBundle\Security\Services\SamlService;
use JMS\Serializer\Serializer;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\Security\Core\Authentication\AuthenticationManagerInterface;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Firewall\LegacyListenerTrait;
use Symfony\Component\Security\Http\Firewall\ListenerInterface;
use Lifestyle\Sso\LaunchpadBundle\Security\Model\UserData\User as UserDto;
use Lifestyle\Sso\LaunchpadBundle\Security\Authentication\Token\SamlUserToken;
use Lifestyle\Sso\LaunchpadBundle\Security\Model\UserApplication\Application as UserApplication;

/**
 * Class SamlListener
 * @package Lifestyle\Sso\LaunchpadBundle\Security\Firewall
 */
class SamlListener implements ListenerInterface
{
    use LegacyListenerTrait;

    /**
     * @var TokenStorageInterface
     */
    protected $tokenStorage;

    /**
     * @var AuthenticationManagerInterface
     */
    protected $authenticationManager;

    /**
     * @var SamlService
     */
    private $samlService;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var SamlConfig;
     */
    private $config;

    /**
     * SamlListener constructor.
     * @param TokenStorageInterface $tokenStorage
     * @param AuthenticationManagerInterface $authenticationManager
     * @param SamlService $samlService
     * @param Serializer $serializer
     * @param SamlConfig $config
     */
    public function __construct(
        TokenStorageInterface $tokenStorage,
        AuthenticationManagerInterface $authenticationManager,
        SamlService $samlService,
        Serializer $serializer,
        SamlConfig $config
    ) {
        $this->tokenStorage = $tokenStorage;
        $this->authenticationManager = $authenticationManager;
        $this->samlService = $samlService;
        $this->serializer = $serializer;
        $this->config = $config;
    }

    /**
     * @param RequestEvent $event
     */
    public function __invoke(RequestEvent $event)
    {
        $saml = $this->samlService->getSimpleSaml();
        if (!$saml->isAuthenticated()) {
            $saml->login();
        }

        $attributes = $saml->getAttributes();
        $user = $this->parseSamlData($attributes);
        $token = $this->mapUserToToken($user);

        try {
            $authToken = $this->authenticationManager->authenticate($token);
            $this->tokenStorage->setToken($authToken);
        } catch (AuthenticationException $e) {
            $token = $this->tokenStorage->getToken();
            if ($token instanceof SamlUserToken) {
                $this->tokenStorage->setToken(null);
            }
            $response = new Response();
            $response->setStatusCode(Response::HTTP_FORBIDDEN);
            $event->setResponse($response);
        }
    }

    /**
     * @param array $samlData
     * @return UserDto
     */
    private function parseSamlData($samlData)
    {
        // init base user data with the SAML response data
        $userDto = new UserDto();
        $userDto->setFromSamlResponse($samlData);

        // parse all given user application data into the DTO
        foreach ($samlData as $key => $entry) {
            if ((strlen($key) > 13) && (substr($key, 0, 13) == 'application::')) {
                $appName = substr($key, 13, 256);
                $appJson = (is_array($entry) && (count($entry) > 0)) ? $entry[0] : $entry;
                try {
                    $appData = $this->serializer->deserialize($appJson, UserApplication::class, 'json');
                } catch (\Exception $exception) {
                    $appData = new UserApplication();
                }
                // now add everything, incl. the original JSON to our user DTO
                $userDto->addUserApplication($appName, $appData);
                $userDto->addUserApplicationJSON($appJson);
            } elseif ($key === 'applications' && is_array($entry)) {
                $userDto->setApplications($entry);
            }
        }
        return $userDto;
    }

    /**
     * @param UserDto $userDto
     * @return SamlUserToken
     */
    private function mapUserToToken(UserDto $userDto)
    {
        $token = new SamlUserToken();

        $token->setUser($userDto->getUsername());
        $token->userName = $userDto->getUsername();
        $token->email = $userDto->getEmail();
        $token->firstname = $userDto->getFirstname();
        $token->lastname = $userDto->getLastname();
        $token->userId = $userDto->getUserId();
        $token->appData = $userDto->getUserApplication($this->config->getIdpAppName());
        $token->userData = $userDto;

        return $token;
    }
}
