import $ from 'jquery';

require('../scss/app.scss');

(function () {
    // Elements
    const $tilesContainer = $('#tiles');
    const $filterContainer = $('#filter');
    const $statusContainer = $('#status');
    const $usernameContainer = $('#username');

    // Classes
    const filterButtonClass = 'btn-primary';
    const filterButtonClassActive = 'btn-light';
    const tileClassHidden = 'deactivated';

    // Configuration
    let appIcons = [];

    // App tile template
    const AppTile = ({name, url, img, title, description, groups}) => `
    <a href="${(!url) ? '#' : url}"
       target="_blank"
       rel="noopener"
       class="card ${(!url) ? 'appurl-missing' : ''} ${(groups) ? groups : ''}">
        <div class="card-body card-body-logo">
            <img src="${img}" alt="">
        </div>
        <div class="card-body card-body-description">
            <h2>${title}</h2>
            <small>${description}</small>
        </div>
    </a>
    `;

    // App filter button template
    const AppFilter = (filter) => `
    <button class="btn ${(filter.selector === '*') ? filterButtonClassActive : filterButtonClass}" 
            data-filter="${(filter.selector === '*') ? '.card' : '.group-' + filter.selector}">${filter.title}</button>
    `;

    /**
     * Get app icon
     *
     * @param {string} selector
     * @return {*}
     */
    function getAppIcon(selector) {
        const fallbackIcon = 'data:image/gif;base64,R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        const selectedIcons = appIcons.filter((appIcon) => appIcon.selector === selector);

        if (selectedIcons.length) {
            return selectedIcons[0]['src'];
        }
        if (selector && selector.length) {
            return selector.replace(/"/g, '\'');
        }

        const defaultIcons = appIcons.filter((icon) => icon.selector === '*');
        return defaultIcons.length > 0 ? defaultIcons[0]['src'] : fallbackIcon;
    }

    /**
     * Create/Update app tiles
     *
     * @param {*} userProfile
     * @param {*} appConfiguration
     */
    function updateApplications(userProfile, appConfiguration) {
        $tilesContainer.find('.card, .loading').remove();
        $tilesContainer.append(userProfile.applications
            .map((appName) => {
                const appConf = appConfiguration.applications[appName] || null;
                return appConf ? {
                    // Internal app name (key)
                    name: appName,
                    // URL or null; Handling of null in template
                    url: appConf.url,
                    // Image or placeholder
                    img: getAppIcon(appConf.img),
                    // Title with app name (key) fallback
                    title: appConf.title || appName,
                    // Description or placeholder
                    description: appConf.description || '&nbsp;',
                    // Group(s)
                    groups: appConf.groups.map((group) => 'group-' + group).join(' ') || null,
                } : null;
            })
            .filter((app) => app !== null)
            .map(AppTile)
            .join(''));
    }

    /**
     * Update user info
     *
     * @param {*} userProfile
     */
    function updateUser(userProfile) {
        const displayName = (typeof userProfile.first_name !== 'undefined'
            && typeof userProfile.last_name !== 'undefined')
            ? `${userProfile.first_name} ${userProfile.last_name}` : userProfile.email;
        $usernameContainer.text(displayName);
    }

    /**
     * Fetch user profile
     *
     * @return {Promise}
     */
    function getUserProfile() {
        return $.getJSON('/user/profile');
    }

    /**
     * Fetch app(s) configuration
     *
     * @return {Promise}
     */
    function getAppConfiguration() {
        return $.getJSON('/configuration');
    }

    /**
     * Add filters to app tiles
     *
     * @param {*} userApplications
     * @param {*} appConfiguration
     */
    function initFilter(userApplications, appConfiguration) {
        // Create one-dimensional array from users application groups
        const userAppGroups = [].concat(...userApplications.map((appName) => {
            const appConf = appConfiguration.applications[appName] || null;
            return null !== appConf ? appConf.groups : [];
        }));

        // Create filters from application groups regarding to users application groups
        const filters = (typeof appConfiguration.application_groups !== 'undefined' ?
            appConfiguration.application_groups : []).filter((filter) => {
            // Skip filters without apps
            const index = Object.values(appConfiguration.applications).findIndex((app) => {
                return typeof app.groups !== 'undefined'
                    && (app.groups.findIndex((group) => group === filter.selector) > -1);
            });
            return index > -1 || filter.selector === '*';
        }).filter((filter) => {
            // Skip filters user is not linked to
            return userAppGroups.indexOf(filter.selector) > -1 || filter.selector === '*';
        });

        // Skip filter creation if all applications belong to the same group
        if (filters.length <= 2) {
            return;
        }

        // Create buttons
        $filterContainer.append(
            filters.map((filter) => {
                return {
                    selector: filter.selector,
                    title: filter.title,
                };
            }).map(AppFilter).join(''),
        );

        $filterContainer.on('click', 'button', (event) => {
            const filterValue = $(event.currentTarget).attr('data-filter');

            // Set button state
            $filterContainer.find('button').removeClass(filterButtonClassActive).addClass(filterButtonClass);
            $(event.currentTarget).removeClass(filterButtonClass).addClass(filterButtonClassActive).blur();

            // Show/hide tiles
            $tilesContainer.find('a').addClass(tileClassHidden).filter(filterValue).removeClass(tileClassHidden);
        });
    }

    /**
     * Deactivate broken links
     */
    function checkLinks() {
        $tilesContainer.on('click', 'a', (event) => {
            $(event.currentTarget).blur();
            return !$(event.currentTarget).hasClass('appurl-missing');
        });
    }

    /**
     * Set status
     *
     * @param {string} status
     */
    function setStatus(status) {
        $statusContainer.text(status);
    }

    /**
     * Check for logged in status
     */
    function ping() {
        $.ajax('/ping', {
            success: () => {
                setStatus('Logged in (' + new Date() + ')');
            },
            error: () => {
                setStatus('Logged out (' + new Date() + ')');
                window.clearInterval(pingInterval);
            },
        });
    }

    // Initial ping
    ping();

    // Check for logged in status every x seconds
    const pingInterval = window.setInterval(ping, 10000);

    // Execute requests, do stuff
    $.when(getUserProfile(), getAppConfiguration()).done((userProfileResponse, appConfigurationResponse) => {
        const userProfile = userProfileResponse[0];
        const appConfiguration = appConfigurationResponse[0];
        const filterMinApplications = (appConfiguration.application_filter
            && appConfiguration.application_filter.min_applications) || 0;
        const filterEnabled = (appConfiguration.application_filter
            && appConfiguration.application_filter.enabled) || false;

        appIcons = appConfiguration.application_icons || [];

        updateUser(userProfile);
        updateApplications(userProfile, appConfiguration);
        checkLinks();

        if (filterEnabled && userProfile.applications.length >= filterMinApplications) {
            initFilter(userProfile.applications, appConfiguration);
        }
    });
})();
