<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\ActivityLogBundle\Database;

use Doctrine\ORM\EntityManagerInterface;
use LifeStyle\ActivityLogBundle\Entity\Factory;
use LifeStyle\ActivityLogBundle\Filter\TextFilter;
use LifeStyle\SamlAuthBundle\Entity\User;

/**
 * Class UserActionLogManager
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    LifeStyle\ActivityLogBundle\Database
 */
class UserActionLogManager
{
    /**
     * @var Factory
     */
    private $factory;

    /**
     * @var EntityManagerInterface
     */
    private $objectManager;

    /**
     * @var TextFilter
     */
    private $textFilter;

    /**
     * UserActionLogManager constructor.
     * @param Factory $factory
     * @param EntityManagerInterface $objectManager
     * @param TextFilter $textFilter
     */
    public function __construct(Factory $factory, EntityManagerInterface $objectManager, TextFilter $textFilter)
    {
        $this->factory = $factory;
        $this->objectManager = $objectManager;
        $this->textFilter = $textFilter;
    }

    /**
     * @param string $controller
     * @param string $action
     * @param string $type
     * @param string $ipAddress
     * @param User $user
     * @param string|null $request
     * @param string|null $response
     * @param array|null $stats
     */
    public function createUserActionLog(
        $controller,
        $action,
        $type,
        $ipAddress,
        User $user,
        $request = null,
        $response = null,
        $stats = []
    ) {
        $userActionLog = $this->factory->userActionLog();
        $userActionLog->setController($controller);
        $userActionLog->setAction($action);
        $userActionLog->setType($type);
        $userActionLog->setIpAddress($ipAddress);
        $userActionLog->setUser($user);
        $userActionLog->setStats((null !== $stats && is_array($stats)) ? $stats : []);

        if (null !== $request || null !== $response) {
            $userActionLogData = $this->factory->userActionLogData();
            if (null !== $request) {
                $userActionLogData->setRequest($this->textFilter->filter($request));
            }
            if (null !== $response) {
                $userActionLogData->setResponse($this->textFilter->filter($response));
            }
            $userActionLog->setUserActionLogData($userActionLogData);
            $this->objectManager->persist($userActionLogData);
        }

        $this->objectManager->persist($userActionLog);
        $this->objectManager->flush();
    }

    /**
     * @return \Doctrine\ORM\Query
     */
    public function getListQuery()
    {
        return $this->objectManager->createQueryBuilder()
            ->select(['l', 'ld'])
            ->from('LifeStyleActivityLogBundle:UserActionLog', 'l')
            ->leftJoin('l.userActionLogData', 'ld')
            ->orderBy('l.createdAt', 'DESC')
            ->getQuery();
    }

    /**
     * @param string $userGuid
     * @return \Doctrine\ORM\Query
     */
    public function getListByUserQuery($userGuid)
    {
        return $this->objectManager->createQueryBuilder()
            ->select(['l', 'ld'])
            ->from('LifeStyleActivityLogBundle:UserActionLog', 'l')
            ->leftJoin('l.userActionLogData', 'ld')
            ->where('l.localUserId = :userGuid')
            ->orderBy('l.createdAt', 'DESC')
            ->setParameter('userGuid', $userGuid)
            ->getQuery();
    }

    /**
     * Remove all log entries
     */
    public function deleteAll()
    {
        $this->objectManager->createQueryBuilder()
            ->delete()
            ->from('LifeStyleActivityLogBundle:UserActionLog', 'l')
            ->getQuery()
            ->execute();
    }
}
