<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\SamlAuthBundle\Model\UserData;

use LifeStyle\SamlAuthBundle\Model\UserApplication\Application;

/**
 * Class User
 * @package LifeStyle\SamlAuthBundle\Model\UserData
 */
final class User
{
    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $userId;

    /**
     * @var string
     */
    private $userGuid;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $firstname;

    /**
     * @var string
     */
    private $lastname;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string[]
     */
    private $userApplicationJSONs;

    /**
     * @var Application[]
     */
    private $userApplications;

    /**
     * @var ServiceUserType
     */
    private $serviceUserType;


    /**
     * User constructor.
     */
    public function __construct()
    {
        $this->serviceUserType = new ServiceUserType();
    }

    /**
     * Set all data fields directly from the SAML response array
     *
     * @param $samlResponse array
     * @return User
     */
    public function setFromSamlResponse($samlResponse)
    {
        $this->userIdentifier = isset($samlResponse['useridentifier'])
            ? $this->normalizeArrayValue($samlResponse['useridentifier']) : '';
        $this->userGuid = isset($samlResponse['guid']) ? $this->normalizeArrayValue($samlResponse['guid']) : '';
        $this->username = isset($samlResponse['username']) ? $this->normalizeArrayValue($samlResponse['username']) : '';
        $this->email = isset($samlResponse['email']) ? $this->normalizeArrayValue($samlResponse['email']) : '';
        $this->firstname = isset($samlResponse['firstname'])
            ? $this->normalizeArrayValue($samlResponse['firstname']) : '';
        $this->lastname = isset($samlResponse['lastname']) ? $this->normalizeArrayValue($samlResponse['lastname']) : '';
    }


    /**
     * Returns the first value/entry of an array, or just the value itself if it is not an array.
     * This is for preventing "index does not exist" as well as being prepared for when maybe at some point in
     * the far future the values are passed as scalars instead of arrays.
     *
     * @param $value mixed
     * @return mixed
     */
    private function normalizeArrayValue($value)
    {
        $returnData = $value;
        if (is_array($value) && (count($value) > 0)) {
            $returnData = $value[0];
        }

        return $returnData;
    }


    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @param string $username
     * @return User
     */
    public function setUsername($username)
    {
        $this->username = $username;

        return $this;
    }

    /**
     * @return string
     */
    public function getUserId()
    {
        return $this->userId;
    }

    /**
     * @param string $userId
     * @return User
     */
    public function setUserId($userId)
    {
        $this->userId = $userId;

        return $this;
    }

    /**
     * @return string
     */
    public function getFirstname()
    {
        return $this->firstname;
    }

    /**
     * @param string $firstname
     * @return User
     */
    public function setFirstname($firstname)
    {
        $this->firstname = $firstname;

        return $this;
    }

    /**
     * @return string
     */
    public function getLastname()
    {
        return $this->lastname;
    }

    /**
     * @param string $lastname
     * @return User
     */
    public function setLastname($lastname)
    {
        $this->lastname = $lastname;

        return $this;
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @param string $email
     * @return User
     */
    public function setEmail($email)
    {
        $this->email = $email;

        return $this;
    }

    /**
     * @return array
     */
    public function getUserApplicationJSONs()
    {
        return $this->userApplicationJSONs;
    }

    /**
     * @param string $userApplicationJSON
     * @return User
     */
    public function setUserApplicationJSONs($userApplicationJSON)
    {
        $this->userApplicationJSONs = $userApplicationJSON;

        return $this;
    }

    /**
     * @param string $json
     * @return User
     */
    public function addUserApplicationJSON($json)
    {
        $this->userApplicationJSONs[] = $json;

        return $this;
    }

    /**
     * @return Application[]
     */
    public function getUserApplications()
    {
        return $this->userApplications;
    }

    /**
     * @return Application
     */
    public function getUserApplication($appName)
    {
        $returnData = null;

        if (isset($this->userApplications[$appName])) {
            $returnData = $this->userApplications[$appName];
        }

        return $returnData;
    }

    /**
     * @param array $userApplications
     * @return User
     */
    public function setUserApplications($userApplications)
    {
        $this->userApplications = $userApplications;

        return $this;
    }

    /**
     * @param string $name
     * @param Application $userApplications
     * @return User
     */
    public function addUserApplication($name, $userApplications)
    {
        $this->userApplications[$name] = $userApplications;

        return $this;
    }

    /**
     * @return string
     */
    public function getUserGuid()
    {
        return $this->userGuid;
    }

    /**
     * @param string $userGuid
     * @return User
     */
    public function setUserGuid($userGuid)
    {
        $this->userGuid = $userGuid;

        return $this;
    }

    /**
     * @return string
     */
    public function getUserIdentifier()
    {
        return $this->userIdentifier;
    }

    /**
     * @param string $userIdentifier
     * @return User
     */
    public function setUserIdentifier($userIdentifier)
    {
        $this->userIdentifier = $userIdentifier;

        return $this;
    }

    /**
     * @return ServiceUserType
     */
    public function getServiceUserType()
    {
        return $this->serviceUserType;
    }

    /**
     * @param ServiceUserType $serviceUserType
     * @return User
     */
    public function setServiceUserType($serviceUserType)
    {
        $this->serviceUserType = $serviceUserType;
        return $this;
    }
}
