<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Tools\RestAuthBundle\Security\Authentication\Provider;

use Sso\Tools\UserIdentifierBundle\Model\Response\UserIdentifier;
use Symfony\Component\DependencyInjection\ContainerInterface as Container;
use Symfony\Component\Security\Core\Authentication\Provider\AuthenticationProviderInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\NonceExpiredException;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\Security\Core\Util\StringUtils;
use LifeStyle\Tools\RestAuthBundle\Api;

/**
 * Class UidentifyProvider
 * @package LifeStyle\Tools\RestAuthBundle\Security\Authentication\Provider
 */
class UidentifyProvider implements AuthenticationProviderInterface
{

    /**
     * Symfony container
     * @var Container
     */
    private $container;

    /**
     * @var UserProviderInterface
     */
    private $userProvider;

    /**
     *
     * @var Api\Manager
     */
    protected $apiM;

    /**
     * @var integer
     */
    protected $wsseLifetime;

    /**
     * @var boolean
     */
    protected $securedAgainstReplayAttack;


    /**
     *
     * @param UserProviderInterface $userProvider
     */
    public function __construct(UserProviderInterface $userProvider, Container $container)
    {
        $this->userProvider = $userProvider;
        $this->container = $container;
        $this->apiM = $this->container->get('uidentify_api_manager');
        $this->wsseLifetime = $this->apiM->configuration()->getWsseLifetime();
        $this->securedAgainstReplayAttack = $this->apiM->configuration()->getSecuredAgainstReplayAttack();
    }

    /**
     *
     * @param TokenInterface $token
     * @return UidentifyToken
     * @throws AuthenticationException
     */
    public function authenticate(TokenInterface $token)
    {
        //first check the servicetoken
        $this->validateServiceToken($token->servicetoken);

        $user = $this->userProvider->loadUserByUsername($token->getUsername());

        if ($user && $this->validateDigest($token->digest, $token->nonce, $token->created, $user->getPassword(),
                $token->getUsername())
        ) {
            $authenticatedToken = new UidentifyToken($user->getRoles());
            $authenticatedToken->setUser($user);
            $user->setRestLastDigest($token->digest);

            return $authenticatedToken;
        }

        throw new AuthenticationException('wrong authentication data', 2);
    }

    /**
     * @param string $serviceToken
     * @return bool
     */
    protected function validateServiceToken($serviceToken)
    {
        $request = $this->container->get('request');
        $service = $request->get('_route');
        //ok check the database
        $tokenEntity = $this->apiM->database()->getTokenRespository()->findOneBy([
            'token' => $serviceToken,
            'service' => $service,
        ]);
        if (empty($tokenEntity) || $tokenEntity === null) {
            throw new AuthenticationException('servicetoken is not valid for: '.$service, 2);
        }

        return true;
    }

    /**
     * This function is specific to Wsse authentication and is only used to help this example
     *
     * For more information specific to the logic here, see
     * https://github.com/symfony/symfony-docs/pull/3134#issuecomment-27699129
     */
    protected function validateDigest($digest, $nonce, $created, $secret, $username)
    {
        // Check created time is not in the future
        if (strtotime($created) > time()) {
            throw new AuthenticationException('x-wsse: created is not valid is in the future ', 2);
        }

        // Expire timestamp after 5 minutes
        if (time() - strtotime($created) > $this->wsseLifetime) {
            throw new AuthenticationException('x-wsse: created is not valid is to long in the past', 2);
        }

        // Validate that the nonce is *not* used in the last 5 minutes
        // if it has, this could be a replay attack
        $checkEntity = $this->apiM->database()->getWsseTokensRespository()->findOneBy(['nonce' => $nonce]);
        if ($checkEntity) {
            throw new NonceExpiredException('Previously used nonce detected');
        }

        // Validate Secret
        $expected = base64_encode(sha1(base64_decode($nonce).$created.$secret, true));
        if (StringUtils::equals($expected, $digest)) {
            if ($this->securedAgainstReplayAttack) {
                //save to db to make header invalid
                $this->apiM->database()->saveWsseToken($nonce, $digest, $created, $username);
            }

            return true;
        }

        return false;
    }

    /**
     * @param TokenInterface $token
     * @return boolean
     */
    public function supports(TokenInterface $token)
    {
        return $token instanceof UidentifyToken;
    }
}
