<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Tools\RestAuthBundle\Security\Firewall;

use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Component\Security\Core\Authentication\AuthenticationManagerInterface;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Firewall\ListenerInterface;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\DependencyInjection\ContainerInterface as Container;
use LifeStyle\Tools\RestAuthBundle\Api\Manager as ApiManager;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApiManager;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as ErrorsIndex;

/**
 * Class UidentifyListener
 * @package LifeStyle\Tools\RestAuthBundle\Security\Firewall
 */
class UidentifyListener implements ListenerInterface
{
    /**
     * @var TokenStorageInterface
     */
    protected $tokenStorage;

    /**
     * @var AuthenticationManagerInterface
     */
    protected $authenticationManager;

    /**
     * @var string
     */
    protected $userIdentifier;

    /**
     *
     * @var Container
     */
    protected $container;

    /**
     *
     * @var ErrorApiManager
     */
    protected $errorApiM;

    /**
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     *
     * @var ErrorsIndex
     */
    protected $errors;

    /**
     * UidentifyListener constructor.
     * @param TokenStorageInterface $tokenStorage
     * @param AuthenticationManagerInterface $authenticationManager
     * @param Container $container
     */
    public function __construct(
        TokenStorageInterface $tokenStorage,
        AuthenticationManagerInterface $authenticationManager,
        Container $container
    ) {

        $this->tokenStorage = $tokenStorage;
        $this->authenticationManager = $authenticationManager;
        $this->container = $container;
        $this->apiM = $this->container->get('uidentify_api_manager');
        $this->errorApiM = $this->container->get('rest_error_api_manager');
        $this->apiM->session()->set('ls_start_microtime', microtime(true));
        $this->errors = $this->errorApiM->error()->errors();
    }

    /**
     * @param GetResponseEvent $event
     * @return type
     */
    public function handle(GetResponseEvent $event)
    {
        $request = $event->getRequest();

        $this->checkHeaderValues($event);

        if ($this->errors->hasErrors()) {
            return $this->getView($event);
        }


        $wsseRegex = '/UsernameToken Username="([^"]+)", PasswordDigest="([^"]+)", Nonce="([^"]+)", Created="([^"]+)"/';
        if (!$request->headers->has('x-wsse') || 1 !== preg_match($wsseRegex, $request->headers->get('x-wsse'),
                $matches)
        ) {
            $this->errors->setStatus(401);
            $this->errors->addError(401, 'x-wsse wrong data in header', 'external');

            if ($this->errors->hasErrors()) {
                return $this->getView($event);
            }
        }

        // here we have to check the user client saml ....
        $token = new UidentifyToken(['ROLE_API']);
        $token->xwsseHeader = $request->headers->get('x-wsse');
        $token->setUser($matches[1]);
        $token->digest = $matches[2];
        $token->nonce = $matches[3];
        $token->created = $matches[4];
        $token->servicetoken = $request->headers->get('servicetoken');

        try {
            $authToken = $this->authenticationManager->authenticate($token);
            $this->tokenStorage->setToken($authToken);
        } catch (AuthenticationException $failed) {
            $this->errors->setStatus(401);
            $this->errors->addError(401, $failed->getMessage(), 'external');

            return $this->getView($event);
        }
    }

    /**
     * @param GetResponseEvent $event
     */
    protected function checkHeaderValues(GetResponseEvent $event)
    {
        $request = $event->getRequest();

        //check the useridentifier
        if (!$request->headers->get('x-wsse')) {
            $this->errors->setStatus(401);
            $this->errors->addError(401, 'x-wsse not found in header', 'external');
        }

        //check the servicetoken
        if (!$request->headers->get('servicetoken')) {
            $this->errors->setStatus(401);
            $this->errors->addError(401, 'servicetoken not found in header', 'external');
        }
    }

    /**
     *
     * @param GetResponseEvent $event
     */
    protected function getView(GetResponseEvent $event)
    {

        $xml = [];
        preg_match("/.xml/i", $event->getRequest()->getUri(), $xml);

        // build the exception response
        if (!empty($xml)) {
            // error response as xml
            $response = $this->errorApiM->error()->response()->response();
            $response->setStatusCode($this->errorApiM->error()->errors()->getStatus());
            $response->headers->set('Content-type', 'text/xml');
            $response->setContent($this->errorApiM->serializer()->serialize($this->errorApiM->error()->errors()->getErrors(),
                'xml'));
        } else {
            // error response as json
            $response = $this->errorApiM->error()->response()->jsonResponse();
            $response->setStatusCode($this->errorApiM->error()->errors()->getStatus());
            $response->setContent($this->errorApiM->serializer()->serialize($this->errorApiM->error()->errors()->getErrors(),
                'json'));
        }

        // Exit with new response set
        $event->setResponse($response);
    }
}
