<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\BackendBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Sso\WebserviceBundle\Entity\Webservice\Type\Application;
use Sso\BackendBundle\Form;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class WsApplicationController
 * @package Sso\BackendBundle\Controller
 */
class WsApplicationController extends WsAbstractController
{
    /**
     *
     * @return Response
     */
    public function indexAction()
    {
        // Prepare pager
        $controller = $this;
        $routeGenerator = function ($page) use ($controller) {
            return $controller->generateUrl('_admin_backend_webservice_application', array('page' => $page));
        };
        list($filterForm, $queryBuilder) = $this->filter(
            new Form\WsApplicationFilterType(),
            $this->appHandler()->getApplicationRepository()
        );
        list($entities, $pagerHtml) = $this->paginator($queryBuilder, $routeGenerator);

        return $this->render('SsoBackendBundle:WsApplication:index.html.twig', array(
            'entities' => $entities,
            'pagerHtml' => $pagerHtml,
            'filterForm' => $filterForm->createView(),
        ));
    }

    /**
     * Ajax search for application
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @return JsonResponse|Response
     */
    public function searchAction(Request $request)
    {
        if (!$request->isXmlHttpRequest()) {
            return new Response('This is no ajax request!', 400);
        }

        $query = $request->get('query', '');
        $limit = $request->get('limit', 10);
        $offset = ($request->get('page', 1) - 1) * $limit;
        $items = array();
        foreach ($this->appHandler()->getApplicationsByName($query, $limit, $offset) as $application) {
            array_push($items, array(
                'id' => $application->getId(),
                'text' => $application->getName(),
            ));
        }

        $response = new JsonResponse(array(
            'total' => count($items),
            'items' => $items,
        ));
        if (null !== ($callback = $request->get('callback'))) {
            $response->setCallback($callback);
        }

        return $response;
    }

    /**
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @return Response
     */
    public function createAction(Request $request)
    {
        $appModel = new Application($this->container->get('validator'));
        $appModel->generateId();
        $form = $this->createCreateForm($appModel);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $this->appHandler()->saveApplication($appModel);
            $this->get('session')->getFlashBag()->add('success', 'Application created success');

            return $this->redirect($this->generateUrl(
                '_admin_backend_webservice_application_show',
                array('applicationId' => $appModel->getId()))
            );
        }

        return $this->render('SsoBackendBundle:WsApplication:new.html.twig', array(
            'entity' => $appModel,
            'form' => $form->createView(),
        ));
    }

    /**
     * Creates a form to create a ApplicationType entity.
     *
     * @param Application $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Application $entity)
    {
        $form = $this->createForm(new Form\WsApplicationType(), $entity, array(
            'action' => $this->generateUrl('_admin_backend_webservice_application_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     *
     * @return Response
     */
    public function newAction()
    {
        $entity = new Application($this->container->get('validator'));
        $form = $this->createCreateForm($entity);

        return $this->render('SsoBackendBundle:WsApplication:new.html.twig', array(
            'entity' => $entity,
            'form' => $form->createView(),
        ));
    }

    /**
     *
     * @param string $applicationId
     * @param int $showUsers
     * @return Response
     * @throws NotFoundHttpException
     */
    public function showAction($applicationId, $showUsers)
    {
        $entity = $this->appHandler()->getApplicationRepository()->find($applicationId);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find WebserviceApplication entity.');
        }

        $deleteForm = $this->createDeleteForm($applicationId);

        return $this->render('SsoBackendBundle:WsApplication:show.html.twig', array(
            'entity' => $entity,
            'delete_form' => $deleteForm->createView(),
            'showUsers' => $showUsers,
        ));
    }

    /**
     *
     * @param string $applicationId
     * @return Response
     * @throws NotFoundHttpException
     */
    public function editAction($applicationId)
    {
        $entity = $this->appHandler()->getApplicationById($applicationId);
        if (!$entity) {
            throw $this->createNotFoundException('Application not found!');
        }

        $editForm = $this->createEditForm($entity);
        $deleteForm = $this->createDeleteForm($applicationId);

        return $this->render('SsoBackendBundle:WsApplication:edit.html.twig', array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
            'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Creates a form to edit a Application entity.
     *
     * @param Application $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createEditForm(Application $entity)
    {
        $form = $this->createForm(new Form\WsApplicationType(), $entity, [
                'action' => $this->generateUrl(
                    '_admin_backend_webservice_application_update',
                    ['applicationId' => $entity->getId()]
                ),
                'method' => 'PUT',
            ]
        );
        if ($form->has('Password')) {
            $form->get('Password')->setData('');
        }

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }

    /**
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param string $applicationId
     * @return Response
     */
    public function updateAction(Request $request, $applicationId)
    {
        $entity = $this->appHandler()->getApplicationById($applicationId);
        if (!$entity) {
            throw $this->createNotFoundException('Application not found!');
        }
        $deleteForm = $this->createDeleteForm($applicationId);
        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            $this->appHandler()->saveApplication($entity);
            $this->userAppHandler()->updateApplication($entity);
            $this->get('session')->getFlashBag()->add('info', 'Application Saved Sucess');

            return $this->redirect(
                $this->generateUrl(
                    '_admin_backend_webservice_application_edit',
                    array('applicationId' => $applicationId)
                )
            );
        }

        return $this->render('SsoBackendBundle:WsApplication:edit.html.twig', array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
            'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     * @param string $applicationId
     * @return Response
     */
    public function deleteAction(Request $request, $applicationId)
    {
        $form = $this->createDeleteForm($applicationId);
        $form->handleRequest($request);

        if ($form->isValid()) {
            $entity = $this->appHandler()->getApplicationById($applicationId);

            if (!$entity) {
                throw $this->createNotFoundException('Unable to find application entity.');
            }

            if ($this->appHandler()->deleteApplication($entity)) {
                $this->addFlash('success', 'Application deleted.');
            } else {
                $this->addFlash(
                    'error',
                    'Error while deleting application! ' .
                    'This may happen if there are users connected to this application. ' .
                    'Disconnect users from application first. '
                );
                if ($form->getData()['force']) {
                    $this->addFlash(
                        'info',
                        'Force-delete function not implemented yet.'
                    );
                }
            }
        } else {
            $this->addFlash('error', 'Error while deleting application! Invalid delete request.');
        }

        return $this->redirect($this->generateUrl('_admin_backend_webservice_application'));
    }

    /**
     * Creates a form to delete a Application entity by id.
     *
     * @param mixed $applicationId The entity id
     *
     * @return \Symfony\Component\Form\Form
     */
    private function createDeleteForm($applicationId)
    {
        return $this->createFormBuilder()
            ->setAction(
                $this->generateUrl(
                    '_admin_backend_webservice_application_delete',
                    array('applicationId' => $applicationId)
                )
            )
            ->setMethod('DELETE')
            ->add('force', 'hidden')
            ->getForm();
    }

    /**
     * @return \Sso\WebserviceBundle\Database\Webservice\Application Database handler for application model
     */
    private function appHandler()
    {
        return $this->databaseManager()->webservice()->application();
    }

    /**
     * @return \Sso\WebserviceBundle\Database\Webservice\UserApplication Database handler for user application model
     */
    private function userAppHandler()
    {
        return $this->databaseManager()->webservice()->userApplication();
    }
}
