<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\PasswordPolicy\Policy;

use Sso\WebserviceBundle\Database\Manager as DatabaseManager;
use Sso\WebserviceBundle\Entity\Webservice\Type\PasswordPolicy as PolicyEntity;
use Sso\WebserviceBundle\Entity\Webservice\Type\PasswordPolicy;

/**
 * Class Handler
 * @package Sso\WebserviceBundle\Api\PasswordPolicy\History
 */
class Service
{
    /**
     * @var DatabaseManager
     */
    private $dbManager;

    /**
     * Service constructor.
     * @param DatabaseManager $dbManager
     */
    public function __construct(DatabaseManager $dbManager)
    {
        $this->dbManager = $dbManager;
    }

    /**
     * @param $userGuid
     * @return Model\Policy
     */
    public function getPolicyForUser($userGuid)
    {
        $userPolicy = null;
        $user = $this->dbManager->webservice()->user()->getUserByGuid($userGuid);

        if (null !== $user) {
            $userPolicy = $this->dbManager->webservice()->passwordPolicy()->getOne($user->getPasswordPolicy());
        }

        if (null === $userPolicy) {
            // for some reason we haven't gotten a policy yet - fetch default policy and work with that
            $userPolicy = $this->dbManager->webservice()->passwordPolicy()->getDefault();
        }

        return $this->policyEntity2Model($userPolicy);
    }

    /**
     * @param $userGuid
     * @return Model\Policy|null
     */
    public function getPolicy($policyId)
    {
        $userPolicy = $this->dbManager->webservice()->passwordPolicy()->getOne($policyId);

        if (null !== $userPolicy) {
            return $this->policyEntity2Model($userPolicy);
        } else {
            return null;
        }
    }

    /**
     * @return Model\Policy|null
     */
    public function getDefaultPolicy()
    {
        $passwordPolicy = $this->dbManager->webservice()->passwordPolicy()->getDefault();
        if (!($passwordPolicy instanceof PasswordPolicy)) {
            return null;
        }
        return $this->policyEntity2Model($passwordPolicy);
    }

    /**
     * @param PolicyEntity $policyEntity
     * @return Model\Policy
     */
    private function policyEntity2Model($policyEntity)
    {
        $policy = new Model\Policy();

        if ($policyEntity instanceof PolicyEntity) {
            $policy
                ->setPolicyId($policyEntity->getPolicyId())
                ->setRegEx($policyEntity->getRegEx())
                ->setValidityDuration($policyEntity->getValidityDuration())
                ->setHistorySize($policyEntity->getHistorySize())
                ->setChangeDelay($policyEntity->getChangeDelay())
                ->setEditDistance($policyEntity->getEditDistance())
                ->setPolicyDescription($policyEntity->getPolicyDescription())
                ->setRegexErrorMessages($policyEntity->getRegexErrorMessages())
                ->setHistorySizeErrorMessages($policyEntity->getHistorySizeErrorMessages())
                ->setEditDistanceErrorMessages($policyEntity->getEditDistanceErrorMessages())
                ->setChangeDelayErrorMessages($policyEntity->getChangeDelayErrorMessages())
                ->setErrorMessages($policyEntity->getErrorMessages());
            //add validations
            foreach ($policyEntity->getPasswordPolicyValidations() as $passwordPolicyValidation) {
                $policyValidationModel = new Model\PolicyValidations();
                $policyValidationModel->setOrderId($passwordPolicyValidation->getOrderId());
                $policyValidationModel->setRegEx($passwordPolicyValidation->getRegEx());
                $policyValidationModel->setErrorMessages($passwordPolicyValidation->getErrorMessages());
                $policy->addPolicyValidation($policyValidationModel);
            }
        } else {
            $policy
                ->setPolicyId('00000000-0000-0000-0000-000000000000')
                ->setRegEx('[A-Za-z0-9]')
                ->setValidityDuration(360)
                ->setHistorySize(1)
                ->setChangeDelay(1)
                ->setEditDistance(1)
                ->setPolicyDescription('')
                ->setRegexErrorMessages([])
                ->setHistorySizeErrorMessages([])
                ->setEditDistanceErrorMessages([])
                ->setChangeDelayErrorMessages([])
                ->setErrorMessages([]);
        }

        return $policy;
    }
}
