<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Entity\Webservice\Type;

use Sso\WebserviceBundle\Api\Error\Type\External;
use Doctrine\Common\Collections\ArrayCollection;
use Symfony\Component\Validator\Validator\ValidatorInterface as Validator;
use Sso\WebserviceBundle\Api\Exception\Type\Api as ApiException;

/**
 * Class Role
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Entity\Webservice\Type
 */
class Role extends Base implements RoleInterface
{
    /**
     * @var string
     */
    private $Id;

    /**
     * @var string
     */
    private $Lowername;

    /**
     * @var string
     */
    private $Name;

    /**
     * @var string
     */
    private $Description;

    /**
     * @var Application
     */
    private $Application;

    /**
     * @var ArrayCollection|UserApplicationRole[]
     */
    private $UserApplicationRoles;
    /**
     * @var string
     */
    private $ApplicationId;

    /**
     * Constructor
     *
     * @param Validator $valApi
     */
    public function __construct(Validator $valApi)
    {
        parent::__construct($valApi);
        $this->UserApplicationRoles = new ArrayCollection();
    }

    /**
     * Initialize object with new values
     *
     * @param array|object $values
     * @param array|null $validationGroups Null for full validation, array of group-name(s) for group specific validation
     */
    public function init($values, $validationGroups = null)
    {
        // Generate new Role-id
        $this->generateId();

        parent::init($values, $validationGroups);
    }

    /**
     * Generate unique role id
     *
     * @throws ApiException
     * @return string
     */
    public function generateId()
    {
        if (null !== $this->Id) {
            throw new ApiException('Changing the application-role-id is not allowed!');
        }
        return $this->Id = hash('sha256', mt_rand() . '#' . microtime());
    }

    /**
     * Get role-id
     *
     * @return string
     */
    public function getId()
    {
        return $this->Id;
    }

    /**
     * Set Id
     *
     * @param string $id
     * @return Role
     */
    public function setId($id)
    {
        $this->Id = $id;

        return $this;
    }

    /**
     * Get role-lowername
     *
     * @return string
     */
    public function getLowername()
    {
        return $this->Lowername;
    }

    /**
     * Get role-name
     *
     * @return string
     */
    public function getName()
    {
        return $this->Name;
    }

    /**
     * Set role-name
     *
     * @param string $Name
     * @return Role
     */
    public function setName($Name)
    {
        $this->Name = $Name;
        $this->Lowername = strtolower($Name);

        return $this;
    }

    /**
     * Get role-description
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->Description;
    }

    /**
     * Set role-description
     *
     * @param string $Description
     * @return Role
     */
    public function setDescription($Description)
    {
        $this->Description = $Description;

        return $this;
    }

    /**
     * Get Application
     *
     * @return Application
     */
    public function getApplication()
    {
        return $this->Application;
    }

    /**
     * Set Application
     *
     * @param Application $Application
     * @return Role
     */
    public function setApplication(Application $Application = null)
    {
        $this->Application = $Application;

        return $this;
    }

    /**
     * Add UserApplicationRoles
     *
     * @param UserApplicationRole $userApplicationRole
     * @return Role
     */
    public function addUserApplicationRole(UserApplicationRole $userApplicationRole)
    {
        $this->UserApplicationRoles[] = $userApplicationRole;

        return $this;
    }

    /**
     * Remove UserApplicationRoles
     *
     * @param UserApplicationRole $userApplicationRole
     */
    public function removeUserApplicationRole(UserApplicationRole $userApplicationRole)
    {
        $this->UserApplicationRoles->removeElement($userApplicationRole);
    }

    /**
     * Get UserApplicationRoles
     *
     * @return UserApplicationRole[]
     */
    public function getUserApplicationRoles()
    {
        return $this->UserApplicationRoles;
    }

    /**
     * Get ApplicationId
     *
     * @return string
     */
    public function getApplicationId()
    {
        return $this->ApplicationId;
    }

    /**
     * Set ApplicationId
     *
     * @param string $applicationId
     * @return Role
     */
    public function setApplicationId($applicationId)
    {
        $this->ApplicationId = $applicationId;

        return $this;
    }

    /**
     * Validate role description
     *
     * @return boolean
     */
    protected function _validateApplication()
    {
        if (!$this->Application->isValid()) {
            $this->errors()->addError(new External('r4', 'Invalid application', 'Role\'s application is not valid.'));
            return false;
        }

        return true;
    }
}
