<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Entity\Webservice\Type;

use Sso\WebserviceBundle\Api\Exception\Type\Api as ApiException;
use Sso\WebserviceBundle\Services\UserApplicationAttribute\Add\Request\AttributeTypeInterface;

/**
 * Class UserApplicationAttribute
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Entity\Webservice\Type
 */
class UserApplicationAttribute extends Base implements AttributeInterface, AttributeTypeInterface
{
    /**
     * @var string
     */
    private $Id;

    /**
     * @var string
     */
    private $Name;

    /**
     * @var string
     */
    private $Type;

    /**
     * @var string
     */
    private $Value;

    /**
     * @var Attribute
     */
    private $Attribute;

    /**
     * @var UserApplication
     */
    private $UserApplication;

    /**
     * @var string
     */
    private $updateHash;

    /**
     * Initialize object with new values
     *
     * @param array|object $values
     * @param array|null $validationGroups Null for full validation, array of group-name(s) for group specific validation
     */
    public function init($values, $validationGroups = null)
    {
        // Generate new user-application-attribute-id
        $this->generateId();

        parent::init($values, $validationGroups);
    }

    /**
     * Set Id
     *
     * @param string $id
     * @throws ApiException
     */
    public function setId($id)
    {
        if (null !== $this->Id && $id !== $this->Id) {
            throw new ApiException('It is not allowed, to change the user-application-attribute-id!');
        }
    }

    /**
     * Get Id
     *
     * @return string
     */
    public function getId()
    {
        return $this->Id;
    }

    /**
     * Generate unique user application attribute id
     *
     * @throws ApiException
     * @return string User-Application-Attribute-Id
     */
    public function generateId()
    {
        if (null !== $this->Id) {
            throw new ApiException('Changing the user-application-attribute-id is not allowed!');
        }
        return $this->Id = hash('sha256', mt_rand() . '#' . microtime());
    }

    /**
     * Set Name
     *
     * @param string $name
     * @return UserApplicationAttribute
     */
    public function setName($name)
    {
        $this->Name = $name;

        return $this;
    }

    /**
     * Get Name
     *
     * @return string
     */
    public function getName(): string
    {
        return (string)$this->Name;
    }

    /**
     * Get lowername
     *
     * @return string
     */
    public function getLowername()
    {
        return strtolower($this->Name);
    }

    /**
     * Set Type
     *
     * @param string $type
     * @return UserApplicationAttribute
     */
    public function setType($type)
    {
        $this->Type = $type;

        return $this;
    }

    /**
     * Get Type
     *
     * @return string
     */
    public function getType(): string
    {
        return $this->Type;
    }

    /**
     * Set Value
     *
     * @param string $value
     * @return UserApplicationAttribute
     */
    public function setValue($value)
    {
        $this->Value = $value;

        return $this;
    }

    /**
     * Get Value
     *
     * @return string
     */
    public function getValue(): string
    {
        return $this->Value;
    }

    /**
     * Set Attribute
     *
     * @param Attribute $attribute
     * @return UserApplicationAttribute
     */
    public function setAttribute(Attribute $attribute = null)
    {
        $this->Attribute = $attribute;
        $this->Name = $attribute->getName();
        $this->Type = $attribute->getType();

        return $this;
    }

    /**
     * Get Attribute
     *
     * @return Attribute
     */
    public function getAttribute()
    {
        return $this->Attribute;
    }

    /**
     * Set UserApplication
     *
     * @param UserApplication $userApplication
     * @return UserApplicationAttribute
     */
    public function setUserApplication(UserApplication $userApplication = null)
    {
        $this->UserApplication = $userApplication;

        return $this;
    }

    /**
     * Get UserApplication
     *
     * @return UserApplication
     */
    public function getUserApplication()
    {
        return $this->UserApplication;
    }

    /**
     * lifecycleCallbacks prePersist
     */
    public function onPrePersist()
    {
        if ($this->getObjectHash() !== $this->updateHash) {
            $this->getUserApplication()->getUser()->setUserUpdatedReferences();
        }
    }

    /**
     * lifecycleCallbacks preUpdate
     */
    public function onPreUpdate()
    {
        if ($this->getObjectHash() !== $this->updateHash) {
            $this->getUserApplication()->getUser()->setUserUpdatedReferences();
        }
    }

    /**
     * lifecycleCallbacks preRemove
     */
    public function onPreRemove()
    {
        $this->getUserApplication()->getUser()->setUserUpdatedReferences();
    }

    /**
     * lifecycleCallbacks postLoad
     */
    public function onPostLoad()
    {
        $this->updateHash = $this->getObjectHash();
    }

    /**
     * @return string
     */
    public function getObjectHash()
    {
        return hash('adler32', $this->Value);
    }
}
