<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Entity\Webservice\Type;

use Sso\WebserviceBundle\Api\Exception\Type\Api as ApiException;

/**
 * Class UserApplicationRole
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Entity\Webservice\Type
 */
class UserApplicationRole extends Base implements RoleInterface
{
    /**
     * @var string
     */
    private $Id;

    /**
     * @var string
     */
    private $Name;

    /**
     * @var boolean
     */
    private $Active;

    /**
     * @var UserApplication
     */
    private $UserApplication;

    /**
     * @var Role
     */
    private $Role;

    /**
     * @var string
     */
    private $RoleId;

    /**
     * @var string
     */
    private $updateHash;

    /**
     * @param array|object $values
     * @param array|null $validationGroups Null for full validation, array of group-name(s) for group specific validation
     * @throws ApiException
     */
    public function init($values, $validationGroups = null)
    {
        // Generate new user-application-role-id
        $this->generateId();

        parent::init($values, $validationGroups);
    }

    /**
     * Generate unique user application role id
     *
     * @throws ApiException
     * @return string User-Application-Role-Id
     */
    public function generateId()
    {
        if (null !== $this->Id) {
            throw new ApiException('Changing the user-application-role-id is not allowed!');
        }
        return $this->Id = hash('sha256', mt_rand() . '#' . microtime());
    }

    /**
     * @return string
     */
    public function getId()
    {
        return $this->Id;
    }

    /**
     * @param string $id
     * @throws ApiException
     */
    public function setId($id)
    {
        if (null !== $this->Id && $id !== $this->Id) {
            throw new ApiException('It is not allowed, to change the user-application-role-id!');
        }
    }

    /**
     * @return boolean
     */
    public function getActive()
    {
        return $this->Active;
    }

    /**
     * @param boolean $active
     * @return UserApplicationRole
     */
    public function setActive($active)
    {
        $this->Active = $active;

        return $this;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->Name;
    }

    /**
     * @param string $Name
     * @return UserApplicationRole
     */
    public function setName($Name)
    {
        $this->Name = $Name;

        return $this;
    }

    /**
     * @return Role
     */
    public function getRole()
    {
        return $this->Role;
    }

    /**
     * @param Role $role
     * @return UserApplicationRole
     */
    public function setRole(Role $role = null)
    {
        $this->Role = $role;
        $this->Name = $role->getName();

        return $this;
    }

    /**
     * @return string
     */
    public function getRoleId()
    {
        return $this->RoleId;
    }

    /**
     * @param string $roleId
     * @return UserApplicationRole
     */
    public function setRoleId($roleId)
    {
        $this->RoleId = $roleId;

        return $this;
    }

    /**
     * @return UserApplication
     */
    public function getUserApplication()
    {
        return $this->UserApplication;
    }

    /**
     * @param UserApplication $userApplication
     * @return UserApplicationRole
     */
    public function setUserApplication(UserApplication $userApplication = null)
    {
        $this->UserApplication = $userApplication;

        return $this;
    }

    /**
     * lifecycleCallbacks prePersist
     */
    public function onPrePersist()
    {
        if ($this->getObjectHash() !== $this->updateHash) {
            $this->getUserApplication()->getUser()->setUserUpdatedReferences();
        }
    }

    /**
     * lifecycleCallbacks preUpdate
     */
    public function onPreUpdate()
    {
        if ($this->getObjectHash() !== $this->updateHash) {
            $this->getUserApplication()->getUser()->setUserUpdatedReferences();
        }
    }

    /**
     * lifecycleCallbacks preRemove
     */
    public function onPreRemove()
    {
        $this->getUserApplication()->getUser()->setUserUpdatedReferences();
    }

    /**
     * lifecycleCallbacks postLoad
     */
    public function onPostLoad()
    {
        $this->updateHash = $this->getObjectHash();
    }

    /**
     * @return string
     */
    public function getObjectHash()
    {
        return hash('adler32', $this->Active ? 1 : 0);
    }
}
