<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\ErrorHandler;

use Sso\WebserviceBundle\ErrorHandler\ErrorStorage\Errors;
use Sso\WebserviceBundle\ErrorHandler\Response\ErrorsInterface;
use JMS\Serializer\SerializerInterface;
use Sso\WebserviceBundle\Api\Error\Type\Base as BaseError;
use Throwable;

/**
 * Class Handler
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\ErrorHandler
 */
final class ErrorHandler implements ErrorHandlerInterface
{
    /**
     * @var Manager
     */
    private $manager;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * Handler constructor.
     * @param Manager $manager
     * @param SerializerInterface $serializer
     */
    public function __construct(Manager $manager, SerializerInterface $serializer)
    {
        $this->manager = $manager;
        $this->serializer = $serializer;
    }

    /**
     * @param int $code
     * @param string $errorCode
     * @param string $errorRef
     * @param string $shortMessage
     * @param string $longMessage
     * @param Throwable|null $exception
     */
    public function addError($code, $errorCode, $errorRef, $shortMessage, $longMessage, Throwable $exception = null)
    {
        $this->manager->errorStorage()->setErrorFromValues($code, $errorCode, $errorRef, $shortMessage, $longMessage, $exception);
    }

    /**
     * @return boolean
     */
    public function hasErrors()
    {
        return $this->manager->errorStorage()->hasErrors();
    }

    /**
     * @return Errors
     */
    public function getErrors()
    {
        return $this->manager->errorStorage()->errors();
    }

    /**
     * @param string $format xml|json
     * @return string
     */
    public function buildResponse($format)
    {
        return $this->serializer->serialize($this->manager->response()->response()->getErrorResponse(), $format);
    }

    /**
     * @return Response\ResponseInterface
     */
    public function getResponseObject(){
        return $this->manager->response()->response()->getErrorResponse();
    }

    /**
     * @return int
     */
    public function getHttpStatusCode()
    {
        $statusCode = $this->manager->response()->response()->getErrors()->getCode();
        return 0 < $statusCode ? $statusCode : 200;
    }

    /**
     * @param array $errors
     */
    public function addErrors(array $errors)
    {
        foreach ($errors as $error) {
            if ($error instanceof BaseError) {
                $this->addErrorFromBaseError($error);
            } else {
                $this->addError(
                    500,
                    '',
                    'eh001',
                    'Invalid error type',
                    sprintf('No error handler for error "%s" available.', get_class($error))
                );
            }
        }
    }

    /**
     * @param BaseError $error
     */
    private function addErrorFromBaseError(BaseError $error)
    {
        $this->addError(
            400,
            $error->code,
            $error->ref,
            $error->shortMessage,
            $error->longMessage
        );
    }
}
