<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\EventListener;

use Sso\WebserviceBundle\Api\ApiVersionInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Sso\WebserviceBundle\ServiceCommand;

/**
 * Class ApiVersionListener
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\EventListener
 */
final class ApiVersionListener implements ApiVersionInterface
{
    const API_VERSION_KEY = 'API-VERSION';
    const API_VERSION_STRING_PREFIX = 'Version';

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var array
     */
    private $validApiVersions = array();

    /**
     * @var int
     */
    private $apiVersion = 1;

    /**
     * @var ServiceCommand
     */
    private $serviceCommand;

    /**
     * ApiVersionListener constructor.
     * @param LoggerInterface $logger
     * @param array $validApiVersions
     */
    public function __construct(LoggerInterface $logger, array $validApiVersions, ServiceCommand $serviceCommand)
    {
        $this->serviceCommand = $serviceCommand;
        $this->logger = $logger;
        foreach ($validApiVersions as $validVersion) {
            $this->addValidVersion($validVersion);
        }
    }

    /**
     * @param GetResponseEvent $event
     */
    public function onKernelRequest(GetResponseEvent $event)
    {
       $this->parseHeader($event);
    }

    /**
     * @param int|string $apiVersion
     */
    private function addValidVersion($apiVersion)
    {
        $apiVersion = (int)$apiVersion;
        if (!in_array($apiVersion, $this->validApiVersions)) {
            $this->validApiVersions[] = $apiVersion;
        }
    }

    /**
     * @param GetResponseEvent $event
     * @throws \HttpInvalidParamException
     */
    public function parseHeader(GetResponseEvent $event)
    {
        $headers = $event->getRequest()->headers;
        if (!$headers->has(self::API_VERSION_KEY)) {
            return;
        }

        $apiVersion = (int)$headers->get(self::API_VERSION_KEY);
        if (!in_array($apiVersion, $this->validApiVersions)) {
            $this->apiVersion = 1;
        } else {
            $this->apiVersion = $apiVersion;
        }

        $this->serviceCommand->setApiVersion($this->apiVersion);
    }

    /**
     * @return int
     */
    public function toInt()
    {
        return $this->apiVersion;
    }

    /**
     * @return string
     */
    public function toString()
    {
        return $this->__toString();
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return self::API_VERSION_STRING_PREFIX . $this->apiVersion;
    }
}
